<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Url;
use Drupal\search_api\IndexInterface;
use Drupal\search_api_sqlite\Index\IndexOperationsInterface;
use Drupal\search_api_sqlite\Utility\IndexSettings;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form for performing SQLite index maintenance operations.
 */
final class IndexOperationsForm extends FormBase {

  /**
   * Constructs an IndexOperationsForm instance.
   *
   * @param \Drupal\search_api_sqlite\Index\IndexOperationsInterface $indexOperations
   *   The index operations service.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   */
  public function __construct(
    private readonly IndexOperationsInterface $indexOperations,
    MessengerInterface $messenger,
  ) {
    $this->setMessenger($messenger);
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public static function create(ContainerInterface $container): static {
    return new self(
      $container->get('search_api_sqlite.index_operations'),
      $container->get('messenger'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'search_api_sqlite_index_operations';
  }

  /**
   * {@inheritdoc}
   *
   * @param array<string, mixed> $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param \Drupal\search_api\IndexInterface|null $search_api_index
   *   The Search API index.
   *
   * @return array<string, mixed>
   *   The form array.
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?IndexInterface $search_api_index = NULL): array {
    if (!$search_api_index instanceof IndexInterface) {
      return $form;
    }

    $index_id = (string) $search_api_index->id();

    // Store the index for submit handler.
    $form_state->set('index', $search_api_index);

    $form['#title'] = $this->t('SQLite Operations: @index', ['@index' => $search_api_index->label()]);

    // Statistics section.
    $form['statistics'] = [
      '#type' => 'details',
      '#title' => $this->t('Index Statistics'),
      '#open' => TRUE,
    ];

    $stats = $this->indexOperations->getStatistics($index_id);

    // Build WAL file status with explanation.
    if ($stats['wal_file_exists']) {
      $wal_status = $this->t('@size (pending writes)', [
        '@size' => $stats['wal_file_size_formatted'],
      ]);
    }
    else {
      $wal_status = $this->t('Empty (checkpointed)');
    }

    $form['statistics']['table'] = [
      '#type' => 'table',
      '#header' => [$this->t('Property'), $this->t('Value')],
      '#rows' => [
        [$this->t('File Path'), $stats['file_path']],
        [$this->t('File Size'), $stats['file_size_formatted']],
        [$this->t('Indexed Items'), number_format($stats['indexed_items'])],
        [$this->t('WAL File'), $wal_status],
      ],
    ];

    if (!empty($stats['tables'])) {
      $form['statistics']['tables'] = [
        '#type' => 'details',
        '#title' => $this->t('Table Details'),
        '#open' => FALSE,
      ];

      $table_rows = [];
      foreach ($stats['tables'] as $table_name => $table_info) {
        $table_rows[] = [$table_name, number_format($table_info['rows'])];
      }

      $form['statistics']['tables']['table'] = [
        '#type' => 'table',
        '#header' => [$this->t('Table'), $this->t('Rows')],
        '#rows' => $table_rows,
      ];
    }

    // Maintenance operations.
    $form['maintenance'] = [
      '#type' => 'details',
      '#title' => $this->t('Maintenance Operations'),
      '#open' => TRUE,
    ];

    // Get index settings for auto-optimize info.
    $settings = IndexSettings::getIndexSettings($search_api_index);
    $auto_optimize = $settings['optimization']['auto_optimize'] ?? FALSE;
    $threshold = $settings['optimization']['optimize_threshold'] ?? 1000;

    $form['maintenance']['optimize'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['form-item']],
    ];

    $optimize_description = $this->t('Merges FTS5 b-tree segments for improved query performance. Safe to run anytime.');
    if ($auto_optimize) {
      $optimize_note = $this->t('Auto-optimize is enabled and runs every @threshold changes.', [
        '@threshold' => number_format($threshold),
      ]);
    }
    else {
      $optimize_note = $this->t('Auto-optimize is disabled. Consider enabling it in the index settings or run manually after heavy write operations.');
    }

    $form['maintenance']['optimize']['description'] = [
      '#markup' => '<p><strong>' . $this->t('Optimize Index') . '</strong><br>' .
      $optimize_description . ' <em>' . $optimize_note . '</em></p>',
    ];

    $form['maintenance']['optimize']['button'] = [
      '#type' => 'submit',
      '#value' => $this->t('Optimize'),
      '#name' => 'optimize',
      '#button_type' => 'primary',
    ];

    $form['maintenance']['vacuum'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['form-item']],
    ];

    $form['maintenance']['vacuum']['description'] = [
      '#markup' => '<p><strong>' . $this->t('Vacuum Database') . '</strong><br>' .
      $this->t('Reclaims disk space and defragments the database file. May take time on large indexes.') . '</p>',
    ];

    $form['maintenance']['vacuum']['button'] = [
      '#type' => 'submit',
      '#value' => $this->t('Vacuum'),
      '#name' => 'vacuum',
    ];

    $form['maintenance']['integrity'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['form-item']],
    ];

    $form['maintenance']['integrity']['description'] = [
      '#markup' => '<p><strong>' . $this->t('Check Integrity') . '</strong><br>' .
      $this->t('Validates FTS5 and SQLite database integrity. Use to diagnose potential corruption.') . '</p>',
    ];

    $form['maintenance']['integrity']['button'] = [
      '#type' => 'submit',
      '#value' => $this->t('Check Integrity'),
      '#name' => 'integrity',
    ];

    $form['maintenance']['rebuild'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['form-item']],
    ];

    $form['maintenance']['rebuild']['description'] = [
      '#markup' => '<p><strong>' . $this->t('Rebuild FTS5 Index') . '</strong><br>' .
      $this->t('Reconstructs the FTS5 internal structures. Use if integrity check fails. Does not require reindexing.') . '</p>',
    ];

    $form['maintenance']['rebuild']['button'] = [
      '#type' => 'submit',
      '#value' => $this->t('Rebuild'),
      '#name' => 'rebuild',
    ];

    // Destructive operations.
    $form['destructive'] = [
      '#type' => 'details',
      '#title' => $this->t('Destructive Operations'),
      '#open' => FALSE,
      '#description' => $this->t('These operations will delete data. Use with caution.'),
    ];

    $form['destructive']['recreate'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['form-item']],
    ];

    $form['destructive']['recreate']['description'] = [
      '#markup' => '<p><strong>' . $this->t('Recreate Index') . '</strong><br>' .
      $this->t('Deletes and recreates the SQLite database file. All indexed data will be lost and all items will be queued for reindexing.') . '</p>',
    ];

    $form['destructive']['recreate']['confirm'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('I understand this will delete all indexed data'),
      '#required' => FALSE,
    ];

    $form['destructive']['recreate']['button'] = [
      '#type' => 'submit',
      '#value' => $this->t('Recreate Index'),
      '#name' => 'recreate',
      '#button_type' => 'danger',
      '#states' => [
        'disabled' => [
          ':input[name="confirm"]' => ['checked' => FALSE],
        ],
      ],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   *
   * @param array<string, mixed> $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  #[\Override]
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    $triggering_element = $form_state->getTriggeringElement();
    $operation = $triggering_element['#name'] ?? '';

    if ($operation === 'recreate' && !$form_state->getValue('confirm')) {
      $form_state->setErrorByName('confirm', $this->t('You must confirm that you understand this will delete all indexed data.'));
    }
  }

  /**
   * {@inheritdoc}
   *
   * @param array<string, mixed> $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $index = $form_state->get('index');
    if (!$index instanceof IndexInterface) {
      $this->messenger()->addError($this->t('Index not found.'));
      return;
    }

    $index_id = (string) $index->id();
    $triggering_element = $form_state->getTriggeringElement();
    $operation = $triggering_element['#name'] ?? '';

    try {
      switch ($operation) {
        case 'optimize':
          $this->indexOperations->optimize($index_id);
          $this->messenger()->addStatus($this->t('Index optimized successfully.'));
          break;

        case 'vacuum':
          $this->indexOperations->vacuum($index_id);
          $this->messenger()->addStatus($this->t('Database vacuumed successfully.'));
          break;

        case 'integrity':
          $result = $this->indexOperations->checkIntegrity($index_id);
          if ($result['valid']) {
            $this->messenger()->addStatus($this->t('Integrity check passed.'));
          }
          else {
            $this->messenger()->addWarning($this->t('Integrity check found issues.'));
          }

          foreach ($result['messages'] as $message) {
            $this->messenger()->addMessage($message);
          }

          break;

        case 'rebuild':
          $this->indexOperations->rebuild($index_id);
          $this->messenger()->addStatus($this->t('FTS5 index rebuilt successfully.'));
          break;

        case 'recreate':
          $this->indexOperations->recreate($index);
          $this->messenger()->addStatus($this->t('Index recreated. All items have been queued for reindexing.'));
          // Redirect to index page after recreate.
          $form_state->setRedirectUrl(Url::fromRoute('entity.search_api_index.canonical', [
            'search_api_index' => $index_id,
          ]));
          break;
      }
    }
    catch (\RuntimeException $runtimeException) {
      $this->messenger()->addError($this->t('Operation failed: @error', ['@error' => $runtimeException->getMessage()]));
    }
  }

}
