<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Plugin\search_api\backend;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Plugin\PluginFormInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\search_api\Attribute\SearchApiBackend;
use Drupal\search_api\Backend\BackendPluginBase;
use Drupal\search_api\Contrib\AutocompleteBackendInterface;
use Drupal\search_api\IndexInterface;
use Drupal\search_api\Item\ItemInterface;
use Drupal\search_api\Query\QueryInterface;
use Drupal\search_api\Query\ResultSetInterface;
use Drupal\search_api\SearchApiException;
use Drupal\search_api_autocomplete\SearchInterface;
use Drupal\search_api_sqlite\Autocomplete\AutocompleteHandlerInterface;
use Drupal\search_api_sqlite\Database\ConnectionManagerInterface;
use Drupal\search_api_sqlite\Database\QueryLoggerInterface;
use Drupal\search_api_sqlite\Database\SchemaManagerInterface;
use Drupal\search_api_sqlite\Enum\MatchingMode;
use Drupal\search_api_sqlite\Enum\Tokenizer;
use Drupal\search_api_sqlite\Index\FieldTypeMapperInterface;
use Drupal\search_api_sqlite\Index\IndexerInterface;
use Drupal\search_api_sqlite\Search\FacetBuilderInterface;
use Drupal\search_api_sqlite\Search\HighlighterInterface;
use Drupal\search_api_sqlite\Search\QueryBuilderInterface;
use Drupal\search_api_sqlite\Utility\ColumnNameHelper;
use Drupal\search_api_sqlite\Utility\IndexSettings;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * SQLite FTS5 search backend implementation.
 */
#[SearchApiBackend(
  id: "search_api_sqlite",
  label: new TranslatableMarkup("SQLite FTS5"),
  description: new TranslatableMarkup("High-performance search backend using SQLite FTS5 for full-text search.")
)]
final class SqliteFts5 extends BackendPluginBase implements ContainerFactoryPluginInterface, PluginFormInterface, AutocompleteBackendInterface {

  /**
   * The connection manager.
   */
  private ConnectionManagerInterface $connectionManager;

  /**
   * The schema manager.
   */
  private SchemaManagerInterface $schemaManager;

  /**
   * The indexer.
   */
  private IndexerInterface $indexer;

  /**
   * The field type mapper.
   */
  private FieldTypeMapperInterface $fieldTypeMapper;

  /**
   * The query builder.
   */
  private QueryBuilderInterface $queryBuilder;

  /**
   * The facet builder.
   */
  private FacetBuilderInterface $facetBuilder;

  /**
   * The autocomplete handler.
   */
  private AutocompleteHandlerInterface $autocompleteHandler;

  /**
   * The highlighter service.
   */
  private HighlighterInterface $highlighter;

  /**
   * The query logger.
   */
  private QueryLoggerInterface $queryLogger;

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ): static {
    $instance = new self($configuration, $plugin_id, $plugin_definition);
    $instance->connectionManager = $container->get('search_api_sqlite.connection_manager');
    $instance->schemaManager = $container->get('search_api_sqlite.schema_manager');
    $instance->indexer = $container->get('search_api_sqlite.indexer');
    $instance->fieldTypeMapper = $container->get('search_api_sqlite.field_type_mapper');
    $instance->queryBuilder = $container->get('search_api_sqlite.query_builder');
    $instance->facetBuilder = $container->get('search_api_sqlite.facet_builder');
    $instance->autocompleteHandler = $container->get('search_api_sqlite.autocomplete_handler');
    $instance->highlighter = $container->get('search_api_sqlite.highlighter');
    $instance->queryLogger = $container->get('search_api_sqlite.query_logger');
    $instance->setLogger($container->get('logger.channel.search_api_sqlite'));
    return $instance;
  }

  /**
   * {@inheritdoc}
   *
   * @return array<string, mixed>
   *   The default configuration.
   */
  #[\Override]
  public function defaultConfiguration(): array {
    return [
      'database_path' => 'private://search_api_sqlite',
      'auto_create_schema' => TRUE,
      'debug_logging' => FALSE,
    ];
  }

  /**
   * Gets the effective configuration for an index.
   *
   * Merges backend configuration with index-level settings.
   *
   * @param \Drupal\search_api\IndexInterface $index
   *   The Search API index.
   *
   * @return array<string, mixed>
   *   The merged configuration.
   */
  public function getIndexConfiguration(IndexInterface $index): array {
    // Get index-level settings merged with defaults.
    $settings = IndexSettings::getIndexSettings($index);

    // Always use backend's database_path.
    $settings['database_path'] = $this->configuration['database_path'] ?? 'private://search_api_sqlite';

    return $settings;
  }

  /**
   * Ensures the connection manager has the correct database path for an index.
   *
   * @param string $index_id
   *   The index ID.
   */
  private function initializeConnection(string $index_id): void {
    $database_path = $this->configuration['database_path'] ?? 'private://search_api_sqlite';
    // This call registers the custom path in the connection manager.
    $this->connectionManager->getConnection($index_id, $database_path);

    // Enable query logging if configured.
    if (!empty($this->configuration['debug_logging'])) {
      $this->queryLogger->enable($index_id);
    }
  }

  /**
   * Ensures the database schema exists for an index.
   *
   * When auto_create_schema is enabled and tables don't exist, this method
   * will create them. This is useful when migrating between environments
   * where the private files directory (containing SQLite databases) is not
   * copied.
   *
   * @param \Drupal\search_api\IndexInterface $index
   *   The Search API index.
   */
  private function ensureTables(IndexInterface $index): void {
    // Skip if auto-create is disabled.
    if (empty($this->configuration['auto_create_schema'])) {
      return;
    }

    $index_id = (string) $index->id();

    // Check if tables already exist.
    if ($this->schemaManager->tablesExist($index_id)) {
      return;
    }

    // Create the schema.
    $config = $this->getIndexConfiguration($index);
    $this->schemaManager->createIndexTables($index, $config);

    $this->getLogger()->info('Auto-created SQLite FTS5 tables for @index (database did not exist).', [
      '@index' => $index_id,
    ]);
  }

  /**
   * {@inheritdoc}
   *
   * @param array<string, mixed> $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array<string, mixed>
   *   The built form.
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form['database_path'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Database directory'),
      '#description' => $this->t('The directory where SQLite database files will be stored. Use stream wrappers like <code>private://search_api_sqlite</code> or an absolute path. Each index will have its own database file in this directory.'),
      '#default_value' => $this->configuration['database_path'] ?? 'private://search_api_sqlite',
      '#required' => TRUE,
    ];

    $form['auto_create_schema'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Auto-create database schema'),
      '#description' => $this->t('When enabled, the SQLite database and schema will be automatically created when indexing starts if they do not exist. This is useful when migrating between environments where the private files directory is not copied. When disabled, indexing will fail if the schema does not exist.'),
      '#default_value' => $this->configuration['auto_create_schema'] ?? TRUE,
    ];

    $form['debug_logging'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable query logging'),
      '#description' => $this->t('When enabled, all SQL queries will be logged to a file in the database directory (e.g., <code>index_name.log</code>). Useful for debugging and performance analysis. <strong>Disable in production.</strong>'),
      '#default_value' => $this->configuration['debug_logging'] ?? FALSE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   *
   * @param array<string, mixed> $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state): void {
    // No additional validation needed.
  }

  /**
   * {@inheritdoc}
   *
   * @param array<string, mixed> $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $this->configuration['database_path'] = $form_state->getValue('database_path') ?? 'private://search_api_sqlite';
    $this->configuration['auto_create_schema'] = (bool) $form_state->getValue('auto_create_schema');
    $this->configuration['debug_logging'] = (bool) $form_state->getValue('debug_logging');
  }

  /**
   * {@inheritdoc}
   *
   * @return array<int, array<string, mixed>>
   *   An array of settings info.
   */
  #[\Override]
  public function viewSettings(): array {
    $settings = [
      [
        'label' => $this->t('Database directory'),
        'info' => $this->configuration['database_path'] ?? 'private://search_api_sqlite',
      ],
      [
        'label' => $this->t('Auto-create schema'),
        'info' => !empty($this->configuration['auto_create_schema']) ? $this->t('Enabled') : $this->t('Disabled'),
      ],
    ];

    if (!empty($this->configuration['debug_logging'])) {
      $settings[] = [
        'label' => $this->t('Query logging'),
        'info' => $this->t('Enabled'),
      ];
    }

    return $settings;
  }

  /**
   * {@inheritdoc}
   */
  public function addIndex(IndexInterface $index): void {
    $index_id = (string) $index->id();
    $this->initializeConnection($index_id);

    $config = $this->getIndexConfiguration($index);
    $this->schemaManager->createIndexTables($index, $config);

    $this->getLogger()->info('Created SQLite FTS5 index tables for @index', [
      '@index' => $index_id,
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function updateIndex(IndexInterface $index): void {
    $index_id = (string) $index->id();
    $this->initializeConnection($index_id);

    $config = $this->getIndexConfiguration($index);
    $needs_reindex = $this->schemaManager->updateIndexSchema($index, $config);

    if ($needs_reindex) {
      $index->reindex();
      $this->getLogger()->info('Index @index schema changed, reindexing required.', [
        '@index' => $index_id,
      ]);
    }
    else {
      $this->getLogger()->info('Updated SQLite FTS5 index schema for @index', [
        '@index' => $index_id,
      ]);
    }
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function removeIndex($index): void {
    $index_id = (string) (is_object($index) ? $index->id() : $index);

    $this->initializeConnection($index_id);
    $this->schemaManager->dropIndexTables($index_id);
    $this->connectionManager->deleteDatabase($index_id);

    $this->getLogger()->info('Removed SQLite FTS5 index @index', [
      '@index' => $index_id,
    ]);
  }

  /**
   * {@inheritdoc}
   *
   * @param \Drupal\search_api\IndexInterface $index
   *   The index to add items to.
   * @param \Drupal\search_api\Item\ItemInterface[] $items
   *   The items to index.
   *
   * @return array<int, string>
   *   The IDs of all items that were successfully indexed.
   */
  public function indexItems(IndexInterface $index, array $items): array {
    $index_id = (string) $index->id();
    $this->initializeConnection($index_id);

    // Ensure schema exists if auto-create is enabled.
    $this->ensureTables($index);

    $config = $this->getIndexConfiguration($index);

    // Pass getSpecialFields as callback so indexer can index special fields
    // (search_api_language, search_api_datasource) without duplicating the
    // logic from BackendPluginBase.
    return $this->indexer->indexItems(
      $index,
      $items,
      $config,
      $this->getSpecialFields(...),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function deleteItems(IndexInterface $index, array $item_ids): void {
    $index_id = (string) $index->id();
    $this->initializeConnection($index_id);
    $this->indexer->deleteItems($index, $item_ids);
  }

  /**
   * {@inheritdoc}
   */
  public function deleteAllIndexItems(IndexInterface $index, $datasource_id = NULL): void {
    $index_id = (string) $index->id();
    $this->initializeConnection($index_id);
    $this->indexer->deleteAllItems($index, $datasource_id);
  }

  /**
   * {@inheritdoc}
   *
   * @return \Drupal\search_api\Query\ResultSetInterface
   *   The search results.
   */
  public function search(QueryInterface $query): ResultSetInterface {
    $results = $query->getResults();
    $index = $query->getIndex();
    $index_id = (string) $index->id();

    // Initialize connection with custom database path.
    $this->initializeConnection($index_id);

    // Get merged configuration for this index.
    $config = $this->getIndexConfiguration($index);

    // Check if tables exist.
    if (!$this->schemaManager->tablesExist($index_id)) {
      $this->getLogger()->warning('Index @index has no tables, returning empty results.', [
        '@index' => $index_id,
      ]);
      return $results;
    }

    try {
      // Build the fulltext field mapping.
      $fulltext_fields = $this->buildFulltextFieldMapping($index);

      // Get matching mode from config.
      $matching_mode = MatchingMode::tryFrom($config['matching'] ?? '') ?? MatchingMode::Words;
      $min_chars = (int) ($config['min_chars'] ?? 1);
      $tokenizer = Tokenizer::tryFrom($config['tokenizer'] ?? '') ?? Tokenizer::Unicode61;

      // Get field boosts for BM25 weighting.
      $field_boosts = $this->schemaManager->getFts5ColumnBoosts($index);

      // Add language condition to the query if languages are specified.
      $this->addLanguageCondition($query);

      // Build and execute the search query.
      $db_query = $this->queryBuilder->buildQuery(
        $query,
        $index_id,
        $fulltext_fields,
        $matching_mode,
        $min_chars,
        $tokenizer,
        $field_boosts,
      );

      // Check for FTS5 highlighting option set by processor.
      $highlight_config = $query->getOption('search_api_sqlite_highlight');
      $has_highlighting = !empty($highlight_config) && !empty($query->getKeys());

      // Add snippet expressions if highlighting is enabled.
      if ($has_highlighting) {
        $fts_table = $this->schemaManager->getFtsTableName($index_id);
        $this->highlighter->addSnippetExpressions(
          $db_query,
          $fts_table,
          $fulltext_fields,
          $highlight_config,
        );
      }

      $statement = $db_query->execute();
      if ($statement === NULL) {
        return $results;
      }

      $rows = $statement->fetchAll();

      // Build result items.
      $fields_helper = $this->getFieldsHelper();

      foreach ($rows as $row) {
        $item_id = $row->item_id;
        $result_item = $fields_helper->createItem($index, $item_id);

        if (isset($row->score)) {
          // BM25 returns negative scores, convert to positive.
          $score = abs((float) $row->score);
          $result_item->setScore($score);
        }

        $results->addResultItem($result_item);
      }

      // Process highlighting if enabled.
      if ($has_highlighting) {
        $this->highlighter->processSnippetResults(
          $results,
          $rows,
          $fulltext_fields,
          $highlight_config,
        );
      }

      // Get total count (without pagination).
      $count_query = clone $db_query;
      $count_query->range();
      $count_statement = $count_query->countQuery()->execute();
      $total_count = $count_statement !== NULL ? (int) $count_statement->fetchField() : 0;
      $results->setResultCount($total_count);

      // Process facets if requested - pass cloned query like search_api_db.
      $facet_options = $query->getOption('search_api_facets');
      if (!empty($facet_options)) {
        $facets = $this->facetBuilder->getFacets(
          $query,
          clone $db_query,
          $index_id,
          $facet_options,
          $fulltext_fields,
          $total_count,
          $field_boosts,
        );
        $results->setExtraData('search_api_facets', $facets);
      }
    }
    catch (\Exception $exception) {
      $this->getLogger()->error('Search failed for index @index: @error', [
        '@index' => $index_id,
        '@error' => $exception->getMessage(),
      ]);
      throw new SearchApiException('Search failed: ' . $exception->getMessage(), 0, $exception);
    }

    return $results;
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function getSupportedFeatures(): array {
    return [
      'search_api_facets',
      'search_api_facets_operator_or',
      'search_api_autocomplete',
      'search_api_random_sort',
      'search_api_spellcheck',
    ];
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function supportsDataType($type): bool {
    return in_array($type, [
      'text',
      'string',
      'integer',
      'decimal',
      'date',
      'boolean',
      'uri',
    ], TRUE);
  }

  /**
   * {@inheritdoc}
   */
  public function getIndexedItemsCount(IndexInterface $index): int {
    $index_id = (string) $index->id();
    $this->initializeConnection($index_id);
    return $this->indexer->getIndexedItemsCount($index);
  }

  /**
   * Builds the fulltext field mapping for an index.
   *
   * @param \Drupal\search_api\IndexInterface $index
   *   The index.
   *
   * @return array<string, string>
   *   Map of field IDs to sanitized column names.
   */
  private function buildFulltextFieldMapping(IndexInterface $index): array {
    $mapping = [];
    $fields = $index->getFields();

    foreach ($fields as $field_id => $field) {
      if ($this->fieldTypeMapper->isFulltextType($field->getType())) {
        $mapping[$field_id] = ColumnNameHelper::sanitize($field_id);
      }
    }

    return $mapping;
  }

  /**
   * {@inheritdoc}
   */
  public function getAutocompleteSuggestions(
    QueryInterface $query,
    SearchInterface $search,
    string $incomplete_key,
    string $user_input,
  ): array {
    $index = $query->getIndex();
    $index_id = (string) $index->id();
    $this->initializeConnection($index_id);

    $fulltext_fields = $this->buildFulltextFieldMapping($index);

    return $this->autocompleteHandler->getSuggestions(
      $query,
      $search,
      $incomplete_key,
      $user_input,
      $fulltext_fields,
    );
  }

  /**
   * Adds language condition to the query if languages are specified.
   *
   * @param \Drupal\search_api\Query\QueryInterface $query
   *   The search query.
   */
  private function addLanguageCondition(QueryInterface $query): void {
    $languages = $query->getLanguages();
    if ($languages !== NULL) {
      $query->addCondition('search_api_language', $languages, 'IN');
    }
  }

  /**
   * {@inheritdoc}
   *
   * @phpstan-ignore-next-line
   */
  protected function getSpecialFields(IndexInterface $index, ?ItemInterface $item = NULL): array {
    $fields = parent::getSpecialFields($index, $item);
    // Remove search_api_id - it's redundant since item_id is already stored
    // in all tables as the primary identifier.
    unset($fields['search_api_id']);
    return $fields;
  }

}
