<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Search;

use Drupal\Core\Database\Query\SelectInterface;
use Drupal\search_api\Query\ResultSetInterface;
use Drupal\search_api_sqlite\Utility\ColumnNameHelper;

/**
 * Provides FTS5 native highlighting using snippet() function.
 *
 * This service adds FTS5 snippet expressions to search queries to enable
 * efficient native highlighting. The snippets are generated as part of the
 * main search query, avoiding additional database round-trips.
 */
final class Highlighter implements HighlighterInterface {

  /**
   * Default highlight configuration.
   *
   * @var array<string, mixed>
   */
  private const array DEFAULTS = [
    'prefix' => '<mark>',
    'suffix' => '</mark>',
    'excerpt_length' => 64,
    'exclude_fields' => [],
  ];

  /**
   * {@inheritdoc}
   */
  public function addSnippetExpressions(
    SelectInterface $db_query,
    string $fts_table,
    array $fulltext_fields,
    array $config,
  ): void {
    $config = array_merge(self::DEFAULTS, $config);
    /** @var array<string> $exclude_fields */
    $exclude_fields = $config['exclude_fields'];

    // FTS5 snippet() syntax:
    // snippet(table, column_index, prefix, suffix, ellipsis, max_tokens)
    // Column index is 0-based, matching the order fields were added to FTS5.
    $column_index = 0;
    foreach (array_keys($fulltext_fields) as $field_id) {
      // Skip excluded fields.
      if (in_array($field_id, $exclude_fields, TRUE)) {
        $column_index++;
        continue;
      }

      // Create unique alias for this snippet.
      $alias = 'snippet_' . ColumnNameHelper::sanitize($field_id);

      // Add snippet expression.
      // Note: We need to use the actual table name, not alias, for FTS5.
      $expression = sprintf(
        "snippet(%s, %d, '%s', '%s', '…', %d)",
        $fts_table,
        $column_index,
        $this->escapeQuotes($config['prefix']),
        $this->escapeQuotes($config['suffix']),
        (int) $config['excerpt_length'],
      );

      $db_query->addExpression($expression, $alias);
      $column_index++;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function processSnippetResults(
    ResultSetInterface $results,
    array $rows,
    array $fulltext_fields,
    array $config,
  ): void {
    $config = array_merge(self::DEFAULTS, $config);
    /** @var array<string> $exclude_fields */
    $exclude_fields = $config['exclude_fields'];
    $result_items = $results->getResultItems();

    // Build a map of item_id => row for quick lookup.
    /** @var array<string, object> $rows_by_id */
    $rows_by_id = [];
    foreach ($rows as $row) {
      /** @var object{item_id: string} $row */
      $rows_by_id[$row->item_id] = $row;
    }

    foreach ($result_items as $item_id => $item) {
      if (!isset($rows_by_id[$item_id])) {
        continue;
      }

      $row = $rows_by_id[$item_id];
      $excerpts = [];

      // Collect snippets from all fields.
      foreach (array_keys($fulltext_fields) as $field_id) {
        if (in_array($field_id, $exclude_fields, TRUE)) {
          continue;
        }

        $alias = 'snippet_' . ColumnNameHelper::sanitize($field_id);
        if (isset($row->$alias) && $row->$alias !== '') {
          $snippet = (string) $row->$alias;
          // Only add if snippet contains highlight markers.
          if (str_contains($snippet, $config['prefix'])) {
            $excerpts[] = $snippet;
          }
        }
      }

      // Combine excerpts and set on item.
      if ($excerpts !== []) {
        $excerpt = implode(' … ', $excerpts);
        $item->setExcerpt($excerpt);
      }
    }
  }

  /**
   * Escapes single quotes for SQL string literals.
   *
   * @param string $value
   *   The value to escape.
   *
   * @return string
   *   The escaped value.
   */
  private function escapeQuotes(string $value): string {
    return str_replace("'", "''", $value);
  }

}
