<?php

declare(strict_types=1);

namespace Drupal\search_api_test_sqlite\EventSubscriber;

use Drupal\Core\State\StateInterface;
use Drupal\search_api_sqlite\Event\FacetsPreExecuteEvent;
use Drupal\search_api_sqlite\Event\MatchQueryAlterEvent;
use Drupal\search_api_sqlite\Event\SearchApiSqliteEvents;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Event subscriber for testing Search API SQLite events.
 *
 * This subscriber can be enabled/disabled via state to allow tests to verify
 * events are dispatched correctly.
 */
final readonly class TestEventSubscriber implements EventSubscriberInterface {

  /**
   * State key for enabling match query alteration.
   */
  public const string STATE_ALTER_MATCH_QUERY = 'search_api_test_sqlite.alter_match_query';

  /**
   * State key for storing the captured match query.
   */
  public const string STATE_CAPTURED_MATCH_QUERY = 'search_api_test_sqlite.captured_match_query';

  /**
   * State key for enabling facets alteration.
   */
  public const string STATE_ALTER_FACETS = 'search_api_test_sqlite.alter_facets';

  /**
   * State key for storing captured facets.
   */
  public const string STATE_CAPTURED_FACETS = 'search_api_test_sqlite.captured_facets';

  /**
   * Constructs a TestEventSubscriber instance.
   *
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   */
  public function __construct(
    private StateInterface $state,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      SearchApiSqliteEvents::MATCH_QUERY_ALTER => 'onMatchQueryAlter',
      SearchApiSqliteEvents::FACETS_PRE_EXECUTE => 'onFacetsPreExecute',
    ];
  }

  /**
   * Handles the MATCH_QUERY_ALTER event.
   *
   * @param \Drupal\search_api_sqlite\Event\MatchQueryAlterEvent $event
   *   The event.
   */
  public function onMatchQueryAlter(MatchQueryAlterEvent $event): void {
    // Always capture the match query for verification.
    $this->state->set(self::STATE_CAPTURED_MATCH_QUERY, $event->getMatchQuery());

    // Optionally alter the query if enabled.
    $alteration = $this->state->get(self::STATE_ALTER_MATCH_QUERY);
    if ($alteration) {
      $event->setMatchQuery($alteration);
    }
  }

  /**
   * Handles the FACETS_PRE_EXECUTE event.
   *
   * @param \Drupal\search_api_sqlite\Event\FacetsPreExecuteEvent $event
   *   The event.
   */
  public function onFacetsPreExecute(FacetsPreExecuteEvent $event): void {
    // Always capture the facets for verification.
    $this->state->set(self::STATE_CAPTURED_FACETS, $event->getFacets());

    // Optionally remove a facet if enabled.
    $facet_to_remove = $this->state->get(self::STATE_ALTER_FACETS);
    if ($facet_to_remove) {
      $event->removeFacet($facet_to_remove);
    }
  }

}
