<?php

declare(strict_types=1);

namespace Drupal\Tests\search_api_sqlite\Kernel;

use Drupal\entity_test\Entity\EntityTestMulRevChanged;
use Drupal\KernelTests\KernelTestBase;
use Drupal\search_api\Entity\Index;
use Drupal\search_api\Entity\Server;
use Drupal\search_api_sqlite\Plugin\search_api\backend\SqliteFts5;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests the auto-create schema functionality of the SQLite FTS5 backend.
 *
 * This tests the scenario where a site is migrated to a new environment
 * (e.g., dev to production) without copying the private files directory,
 * which contains the SQLite database files. When indexing starts, the
 * backend should automatically create the database and schema if the
 * auto_create_schema option is enabled.
 *
 * @group search_api_sqlite
 */
#[CoversClass(SqliteFts5::class)]
#[Group('search_api_sqlite')]
class AutoCreateSchemaTest extends KernelTestBase {

  use SqliteDatabaseCleanupTrait;

  /**
   * {@inheritdoc}
   *
   * @var list<string>
   */
  protected static $modules = [
    'field',
    'search_api',
    'search_api_sqlite',
    'search_api_test',
    'entity_test',
    'text',
    'user',
    'system',
  ];

  /**
   * The test server.
   */
  protected Server $server;

  /**
   * The test index.
   */
  protected Index $index;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('entity_test_mulrev_changed');
    $this->installEntitySchema('search_api_task');
    $this->installSchema('search_api', ['search_api_item']);
    $this->installConfig(['search_api', 'search_api_test']);

    // Create server with auto_create_schema enabled (default).
    $this->server = Server::create([
      'id' => 'test_server',
      'name' => 'Test Server',
      'backend' => 'search_api_sqlite',
      'backend_config' => [
        'database_path' => $this->getTestDatabasePath(),
        'auto_create_schema' => TRUE,
        'debug_logging' => FALSE,
      ],
    ]);
    $this->server->save();

    // Create index but do NOT add it to the server yet.
    // This simulates starting with a fresh database.
    $this->index = Index::create([
      'id' => 'test_index',
      'name' => 'Test Index',
      'server' => 'test_server',
      'datasource_settings' => [
        'entity:entity_test_mulrev_changed' => [],
      ],
      'field_settings' => [
        'name' => [
          'label' => 'Name',
          'type' => 'text',
          'datasource_id' => 'entity:entity_test_mulrev_changed',
          'property_path' => 'name',
        ],
      ],
    ]);
    $this->index->save();
  }

  /**
   * Tests that schema is auto-created when indexing with enabled option.
   */
  public function testAutoCreateSchemaOnIndexing(): void {
    $backend = $this->server->getBackend();
    $this->assertInstanceOf(SqliteFts5::class, $backend);

    // Get the connection manager to check database existence.
    $connection_manager = \Drupal::service('search_api_sqlite.connection_manager');
    $schema_manager = \Drupal::service('search_api_sqlite.schema_manager');

    // Delete the database if it exists (simulate fresh environment).
    $connection_manager->deleteDatabase('test_index');

    // Verify database does not exist.
    $this->assertFalse(
      $connection_manager->databaseExists('test_index'),
      'Database should not exist before indexing.'
    );

    // Create a test entity to index.
    $entity = $this->createTestEntity('Test Item');

    // Get items to index.
    $item_ids = ['entity:entity_test_mulrev_changed/' . $entity->id() . ':en'];
    $items = [];
    foreach ($item_ids as $item_id) {
      $items[$item_id] = \Drupal::service('search_api.fields_helper')
        ->createItemFromObject(
          $this->index,
          $entity->getTypedData(),
          $item_id,
        );
    }

    // Index items - this should auto-create the schema.
    $indexed = $backend->indexItems($this->index, $items);

    // Verify the item was indexed.
    $this->assertNotEmpty($indexed, 'Item should be indexed.');

    // Verify database now exists.
    $this->assertTrue(
      $connection_manager->databaseExists('test_index'),
      'Database should exist after indexing with auto_create_schema enabled.'
    );

    // Verify tables exist.
    $this->assertTrue(
      $schema_manager->tablesExist('test_index'),
      'Tables should exist after auto-creation.'
    );
  }

  /**
   * Tests that schema is NOT auto-created when option is disabled.
   */
  public function testNoAutoCreateSchemaWhenDisabled(): void {
    // Update server to disable auto_create_schema.
    $config = $this->server->getBackendConfig();
    $config['auto_create_schema'] = FALSE;
    $this->server->setBackendConfig($config);
    $this->server->save();

    $backend = $this->server->getBackend();
    $this->assertInstanceOf(SqliteFts5::class, $backend);

    $connection_manager = \Drupal::service('search_api_sqlite.connection_manager');

    // Delete the database if it exists.
    $connection_manager->deleteDatabase('test_index');

    // Verify database does not exist.
    $this->assertFalse(
      $connection_manager->databaseExists('test_index'),
      'Database should not exist before indexing.'
    );

    // Create a test entity to index.
    $entity = $this->createTestEntity('Test Item');

    // Get items to index.
    $item_ids = ['entity:entity_test_mulrev_changed/' . $entity->id() . ':en'];
    $items = [];
    foreach ($item_ids as $item_id) {
      $items[$item_id] = \Drupal::service('search_api.fields_helper')
        ->createItemFromObject(
          $this->index,
          $entity->getTypedData(),
          $item_id,
        );
    }

    // Indexing should fail because schema doesn't exist and auto-create is off.
    $this->expectException(\Exception::class);
    $backend->indexItems($this->index, $items);
  }

  /**
   * Tests that existing schema is not recreated on subsequent indexing.
   */
  public function testSchemaNotRecreatedWhenExists(): void {
    $backend = $this->server->getBackend();
    $this->assertInstanceOf(SqliteFts5::class, $backend);

    $connection_manager = \Drupal::service('search_api_sqlite.connection_manager');
    $schema_manager = \Drupal::service('search_api_sqlite.schema_manager');

    // Ensure fresh start.
    $connection_manager->deleteDatabase('test_index');

    // Create first entity and index it.
    $entity1 = $this->createTestEntity('First Item');
    $item_id1 = 'entity:entity_test_mulrev_changed/' . $entity1->id() . ':en';
    $items1 = [
      $item_id1 => \Drupal::service('search_api.fields_helper')
        ->createItemFromObject(
          $this->index,
          $entity1->getTypedData(),
          $item_id1,
      ),
    ];

    $indexed1 = $backend->indexItems($this->index, $items1);
    $this->assertNotEmpty($indexed1, 'First item should be indexed.');

    // Create second entity and index it.
    $entity2 = $this->createTestEntity('Second Item');
    $item_id2 = 'entity:entity_test_mulrev_changed/' . $entity2->id() . ':en';
    $items2 = [
      $item_id2 => \Drupal::service('search_api.fields_helper')
        ->createItemFromObject(
          $this->index,
          $entity2->getTypedData(),
          $item_id2,
      ),
    ];

    $indexed2 = $backend->indexItems($this->index, $items2);
    $this->assertNotEmpty($indexed2, 'Second item should be indexed.');

    // Verify both items exist in the database (schema wasn't recreated).
    $connection = $connection_manager->getConnection('test_index');
    $items_table = $schema_manager->getItemsTableName('test_index');
    $result = $connection->select($items_table, 'i')
      ->countQuery()
      ->execute();
    $count = $result !== NULL ? (int) $result->fetchField() : 0;

    $this->assertEquals(2, $count, 'Both items should exist (schema was not recreated).');
  }

  /**
   * Creates a test entity.
   *
   * @param string $name
   *   The entity name.
   *
   * @return \Drupal\entity_test\Entity\EntityTestMulRevChanged
   *   The created entity.
   */
  protected function createTestEntity(string $name): EntityTestMulRevChanged {
    $entity = EntityTestMulRevChanged::create([
      'name' => $name,
    ]);
    $entity->save();
    return $entity;
  }

}
