<?php

declare(strict_types=1);

namespace Drupal\Tests\search_api_sqlite\Kernel;

use Symfony\Component\Yaml\Yaml;
use Drupal\search_api\Entity\Server;
use PHPUnit\Framework\Attributes\After;
use PHPUnit\Framework\Attributes\Before;

/**
 * Provides unique database paths and cleanup for SQLite in parallel tests.
 *
 * This trait solves the parallel test execution problem by generating unique
 * database paths for each test method. It uses a combination of:
 * - TEST_TOKEN env var (set by PHPUnit --process-isolation or paratest)
 * - Class name hash
 * - Unique ID per test method.
 *
 * Use this trait in kernel tests: `use SqliteDatabaseCleanupTrait;`
 *
 * If your test class has a `$serverId` property, call configureSqliteTestServer
 * in your setUp() after installConfig() to update the server configuration.
 */
trait SqliteDatabaseCleanupTrait {

  /**
   * Unique database path for this test instance.
   */
  protected string $sqliteTestDatabasePath;

  /**
   * Sets up unique database path before each test.
   *
   * @before
   */
  #[Before]
  public function setUpSqliteTestDatabase(): void {
    $this->sqliteTestDatabasePath = $this->generateUniqueDatabasePath();
  }

  /**
   * Cleans up SQLite database after each test.
   *
   * @after
   */
  #[After]
  public function tearDownSqliteTestDatabase(): void {
    if (isset($this->sqliteTestDatabasePath)) {
      $real_path = $this->resolveDatabasePath($this->sqliteTestDatabasePath);
      if ($real_path && is_dir($real_path)) {
        $this->cleanupDirectory($real_path);
        @rmdir($real_path);
      }
    }
  }

  /**
   * Configures the test server to use a unique database path.
   *
   * IMPORTANT: Call this BEFORE installing index config to ensure the database
   * is created at the correct path. The order should be:
   * 1. Install server config (via installConfig or createSqliteTestServer)
   * 2. Configure unique path: configureSqliteTestServer()
   * 3. Install index config.
   *
   * @param string|null $server_id
   *   Server ID. If not provided, uses $this->serverId if defined.
   *
   * @phpstan-ignore function.alreadyNarrowedType
   */
  protected function configureSqliteTestServer(?string $server_id = NULL): void {
    // Use provided server_id, or fall back to $this->serverId via property.
    // @phpstan-ignore function.alreadyNarrowedType
    if ($server_id === NULL && property_exists($this, 'serverId')) {
      /** @phpstan-ignore property.notFound */
      $server_id = $this->serverId;
    }

    if ($server_id === NULL) {
      return;
    }

    $server = Server::load($server_id);
    if ($server === NULL) {
      return;
    }

    $config = $server->getBackendConfig();
    $config['database_path'] = $this->getTestDatabasePath();
    $server->setBackendConfig($config);
    $server->save();
  }

  /**
   * Installs server and index configs with unique database path.
   *
   * This method handles the correct order of operations to ensure the SQLite
   * database is created at a unique path for parallel test execution:
   * 1. Import server config from module
   * 2. Update server to use unique database path
   * 3. Import index config (with updated server path)
   *
   * @param string $server_config_name
   *   The server config name (e.g., 'search_api.server.my_server').
   * @param string $index_config_name
   *   The index config name (e.g., 'search_api.index.my_index').
   * @param string $module
   *   The module containing the config.
   * @param string[] $additional_modules
   *   Additional modules to install config from after server is configured.
   */
  protected function installSqliteTestConfig(
    string $server_config_name,
    string $index_config_name,
    string $module,
    array $additional_modules = [],
  ): void {
    // Import just the server config first.
    $this->importConfigFromModule($server_config_name, $module);

    // Extract server ID from config name.
    $server_id = str_replace('search_api.server.', '', $server_config_name);

    // Update server to use unique path before index is created.
    $this->configureSqliteTestServer($server_id);

    // Now import the index config.
    // Note: Don't use installConfig() - it would recreate the server too.
    $this->importConfigFromModule($index_config_name, $module);

    // Install any additional modules.
    if (!empty($additional_modules)) {
      $this->installConfig($additional_modules);
    }
  }

  /**
   * Imports a single config entity from a module's config/install.
   *
   * @param string $config_name
   *   The config name to import.
   * @param string $module
   *   The module containing the config.
   */
  protected function importConfigFromModule(string $config_name, string $module): void {
    $module_path = \Drupal::service('extension.list.module')->getPath($module);
    $config_path = $module_path . '/config/install/' . $config_name . '.yml';

    if (!file_exists($config_path)) {
      throw new \RuntimeException("Config file not found: $config_path");
    }

    $data = Yaml::parseFile($config_path);

    // For config entities, we need to create them via the entity system.
    // Extract entity type from config name (e.g., 'search_api.server.xxx').
    $parts = explode('.', $config_name);
    if (count($parts) >= 3 && $parts[0] === 'search_api') {
      $entity_type = 'search_api_' . $parts[1];
      $storage = \Drupal::entityTypeManager()->getStorage($entity_type);
      $entity = $storage->create($data);
      $entity->save();
    }
    else {
      // Simple config.
      \Drupal::configFactory()->getEditable($config_name)->setData($data)->save();
    }
  }

  /**
   * Generates a unique database path for parallel-safe test execution.
   *
   * @return string
   *   A unique temporary:// path for this test instance.
   */
  protected function generateUniqueDatabasePath(): string {
    // Use TEST_TOKEN if available (paratest/PHPUnit parallel).
    $token = getenv('TEST_TOKEN') ?: '';

    // Include class name for additional uniqueness.
    $class_hash = substr(md5(static::class), 0, 8);

    // Add unique ID for this specific test run.
    $unique_id = uniqid('', TRUE);

    $path_parts = array_filter([
      'search_api_sqlite_test',
      $token,
      $class_hash,
      str_replace('.', '_', $unique_id),
    ]);

    return 'temporary://' . implode('_', $path_parts);
  }

  /**
   * Gets the unique database path for this test.
   *
   * @return string
   *   The unique database path.
   */
  protected function getTestDatabasePath(): string {
    return $this->sqliteTestDatabasePath;
  }

  /**
   * Resolves a stream wrapper path to a real filesystem path.
   *
   * @param string $path
   *   The stream wrapper path (e.g., temporary://...).
   *
   * @return string|null
   *   The real path, or NULL if it cannot be resolved.
   */
  protected function resolveDatabasePath(string $path): ?string {
    if (str_starts_with($path, 'temporary://')) {
      $relative = substr($path, strlen('temporary://'));
      if (function_exists('file_directory_temp')) {
        return file_directory_temp() . '/' . $relative;
      }
      return sys_get_temp_dir() . '/' . $relative;
    }
    return $path;
  }

  /**
   * Removes all SQLite files from a directory.
   *
   * @param string $directory
   *   The directory path to clean.
   */
  protected function cleanupDirectory(string $directory): void {
    if (!is_dir($directory)) {
      return;
    }

    $files = glob($directory . '/*.sqlite*') ?: [];
    foreach ($files as $file) {
      @unlink($file);
    }
  }

}
