<?php

declare(strict_types=1);

namespace Drupal\Tests\search_api_sqlite\Unit\Index;

use Drupal\search_api\Item\FieldInterface;
use Drupal\search_api\Utility\DataTypeHelperInterface;
use Drupal\search_api_sqlite\Index\FieldTypeMapper;
use Drupal\search_api_sqlite\Index\FieldTypeMapperInterface;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\DataProvider;
use PHPUnit\Framework\Attributes\Group;
use PHPUnit\Framework\MockObject\MockObject;

/**
 * Tests the FieldTypeMapper service.
 *
 * @group search_api_sqlite
 */
#[CoversClass(FieldTypeMapper::class)]
#[Group('search_api_sqlite')]
final class FieldTypeMapperTest extends UnitTestCase {

  /**
   * The field type mapper under test.
   */
  private FieldTypeMapper $mapper;

  /**
   * The mocked data type helper.
   */
  private MockObject&DataTypeHelperInterface $dataTypeHelper;

  /**
   * {@inheritdoc}
   */
  #[\Override]
  protected function setUp(): void {
    parent::setUp();

    $this->dataTypeHelper = $this->createMock(DataTypeHelperInterface::class);
    $this->dataTypeHelper->method('isTextType')
      ->willReturnCallback(fn(string $type): bool => $type === 'text');

    $this->mapper = new FieldTypeMapper($this->dataTypeHelper);
  }

  /**
   * Tests isFulltextType() with various types.
   *
   * @dataProvider fulltextTypeProvider
   */
  #[DataProvider('fulltextTypeProvider')]
  public function testIsFulltextType(string $type, bool $expected): void {
    $this->assertSame($expected, $this->mapper->isFulltextType($type));
  }

  /**
   * Data provider for fulltext type tests.
   *
   * @return array<string, array{string, bool}>
   *   Test cases.
   */
  public static function fulltextTypeProvider(): array {
    return [
      'text is fulltext' => ['text', TRUE],
      'string is not fulltext' => ['string', FALSE],
      'integer is not fulltext' => ['integer', FALSE],
      'date is not fulltext' => ['date', FALSE],
      'boolean is not fulltext' => ['boolean', FALSE],
      'decimal is not fulltext' => ['decimal', FALSE],
      'uri is not fulltext' => ['uri', FALSE],
    ];
  }

  /**
   * Tests getStorageType() with various types.
   *
   * @dataProvider storageTypeProvider
   */
  #[DataProvider('storageTypeProvider')]
  public function testGetStorageType(string $type, string $expected): void {
    $this->assertSame($expected, $this->mapper->getStorageType($type));
  }

  /**
   * Data provider for storage type tests.
   *
   * @return array<string, array{string, string}>
   *   Test cases.
   */
  public static function storageTypeProvider(): array {
    return [
      'text maps to fts5' => ['text', FieldTypeMapperInterface::STORAGE_FTS5],
      'string maps to string' => ['string', FieldTypeMapperInterface::STORAGE_STRING],
      'uri maps to string' => ['uri', FieldTypeMapperInterface::STORAGE_STRING],
      'integer maps to integer' => ['integer', FieldTypeMapperInterface::STORAGE_INTEGER],
      'boolean maps to integer' => ['boolean', FieldTypeMapperInterface::STORAGE_INTEGER],
      'date maps to integer' => ['date', FieldTypeMapperInterface::STORAGE_INTEGER],
      'decimal maps to decimal' => ['decimal', FieldTypeMapperInterface::STORAGE_DECIMAL],
      'unknown maps to string' => ['unknown_type', FieldTypeMapperInterface::STORAGE_STRING],
    ];
  }

  /**
   * Tests convertValue() for text type.
   */
  public function testConvertValueText(): void {
    $this->assertSame('hello world', $this->mapper->convertValue('hello world', 'text'));
  }

  /**
   * Tests convertValue() strips HTML from text.
   */
  public function testConvertValueTextStripsHtml(): void {
    $this->assertSame('hello world', $this->mapper->convertValue('<p>hello</p> <b>world</b>', 'text'));
  }

  /**
   * Tests convertValue() normalizes whitespace in text.
   */
  public function testConvertValueTextNormalizesWhitespace(): void {
    $this->assertSame('hello world', $this->mapper->convertValue("hello\n\t  world", 'text'));
  }

  /**
   * Tests convertValue() for string type.
   */
  public function testConvertValueString(): void {
    $this->assertSame('test string', $this->mapper->convertValue('test string', 'string'));
  }

  /**
   * Tests convertValue() truncates long strings.
   */
  public function testConvertValueStringTruncatesLong(): void {
    $longString = str_repeat('a', 2000);
    $result = $this->mapper->convertValue($longString, 'string');
    $this->assertSame(1024, strlen((string) $result));
  }

  /**
   * Tests convertValue() for integer type.
   */
  public function testConvertValueInteger(): void {
    $this->assertSame(42, $this->mapper->convertValue(42, 'integer'));
    $this->assertSame(42, $this->mapper->convertValue('42', 'integer'));
  }

  /**
   * Tests convertValue() for boolean type.
   */
  public function testConvertValueBoolean(): void {
    $this->assertSame(1, $this->mapper->convertValue(TRUE, 'boolean'));
    $this->assertSame(0, $this->mapper->convertValue(FALSE, 'boolean'));
    $this->assertSame(1, $this->mapper->convertValue(1, 'boolean'));
    $this->assertSame(0, $this->mapper->convertValue(0, 'boolean'));
  }

  /**
   * Tests convertValue() for date type with timestamp.
   */
  public function testConvertValueDateTimestamp(): void {
    $timestamp = 1701936000;
    $this->assertSame($timestamp, $this->mapper->convertValue($timestamp, 'date'));
  }

  /**
   * Tests convertValue() for date type with date string.
   */
  public function testConvertValueDateString(): void {
    $result = $this->mapper->convertValue('2023-12-07', 'date');
    $this->assertIsInt($result);
    $this->assertGreaterThan(0, $result);
  }

  /**
   * Tests convertValue() for decimal type.
   */
  public function testConvertValueDecimal(): void {
    $this->assertSame(3.14, $this->mapper->convertValue(3.14, 'decimal'));
    $this->assertSame(3.14, $this->mapper->convertValue('3.14', 'decimal'));
  }

  /**
   * Tests convertValue() returns NULL for NULL values.
   */
  public function testConvertValueNullReturnsNull(): void {
    $this->assertNull($this->mapper->convertValue(NULL, 'string'));
    $this->assertNull($this->mapper->convertValue(NULL, 'integer'));
    $this->assertNull($this->mapper->convertValue(NULL, 'text'));
  }

  /**
   * Tests convertValue() returns NULL for empty string.
   */
  public function testConvertValueEmptyStringReturnsNull(): void {
    $this->assertNull($this->mapper->convertValue('', 'string'));
    $this->assertNull($this->mapper->convertValue('', 'integer'));
    $this->assertNull($this->mapper->convertValue('', 'text'));
  }

  /**
   * Tests getFieldDataColumn() returns correct column names.
   *
   * @dataProvider fieldDataColumnProvider
   */
  #[DataProvider('fieldDataColumnProvider')]
  public function testGetFieldDataColumn(string $storageType, string $expected): void {
    $this->assertSame($expected, $this->mapper->getFieldDataColumn($storageType));
  }

  /**
   * Data provider for field data column tests.
   *
   * @return array<string, array{string, string}>
   *   Test cases.
   */
  public static function fieldDataColumnProvider(): array {
    return [
      'string storage' => [FieldTypeMapperInterface::STORAGE_STRING, 'value_string'],
      'integer storage' => [FieldTypeMapperInterface::STORAGE_INTEGER, 'value_integer'],
      'decimal storage' => [FieldTypeMapperInterface::STORAGE_DECIMAL, 'value_decimal'],
      'unknown defaults to string' => ['unknown', 'value_string'],
    ];
  }

  /**
   * Tests extractFieldValues() extracts and converts values.
   */
  public function testExtractFieldValues(): void {
    $field = $this->createMock(FieldInterface::class);
    $field->method('getValues')->willReturn(['value1', 'value2']);
    $field->method('getType')->willReturn('string');

    $result = $this->mapper->extractFieldValues($field);

    $this->assertSame(['value1', 'value2'], $result);
  }

  /**
   * Tests extractFieldValues() filters out NULL values.
   */
  public function testExtractFieldValuesFiltersNull(): void {
    $field = $this->createMock(FieldInterface::class);
    $field->method('getValues')->willReturn(['value1', '', NULL, 'value2']);
    $field->method('getType')->willReturn('string');

    $result = $this->mapper->extractFieldValues($field);

    $this->assertSame(['value1', 'value2'], $result);
  }

  /**
   * Tests extractFieldValues() returns empty array for empty field.
   */
  public function testExtractFieldValuesEmptyField(): void {
    $field = $this->createMock(FieldInterface::class);
    $field->method('getValues')->willReturn([]);
    $field->method('getType')->willReturn('string');

    $result = $this->mapper->extractFieldValues($field);

    $this->assertSame([], $result);
  }

}
