<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Database;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Database\Database;
use Drupal\Core\File\FileSystemInterface;
use Psr\Log\LoggerInterface;

/**
 * Manages SQLite database connections for Search API indexes.
 */
final class ConnectionManager implements ConnectionManagerInterface {

  /**
   * The connection key prefix.
   */
  private const string CONNECTION_PREFIX = 'search_api_sqlite_';

  /**
   * Maximum number of item IDs per INSERT batch.
   *
   * Used when populating the temp table with item IDs.
   * Higher values reduce INSERT overhead but use more memory.
   */
  private const int INSERT_BATCH_SIZE = 500;

  /**
   * Threshold for switching from IN() clause to temp table approach.
   *
   * Below this threshold, use IN() clause (faster for small sets).
   * At or above this threshold, use temp table (more efficient for large sets).
   * SQLite has a default SQLITE_MAX_VARIABLE_NUMBER of 999.
   */
  private const int TEMP_TABLE_THRESHOLD = 500;

  /**
   * Custom database paths per index.
   *
   * @var array<string, string>
   */
  private array $customPaths = [];

  /**
   * Constructs a ConnectionManager instance.
   *
   * @param \Drupal\Core\File\FileSystemInterface $fileSystem
   *   The file system service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger.
   */
  public function __construct(
    private readonly FileSystemInterface $fileSystem,
    private readonly ConfigFactoryInterface $configFactory,
    private readonly LoggerInterface $logger,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function getConnection(string $index_id, ?string $database_path = NULL): Connection {
    $connection_key = $this->getConnectionKey($index_id);

    // Store custom path if provided.
    if ($database_path !== NULL) {
      $this->customPaths[$index_id] = $database_path;
    }

    // Check if connection already exists.
    if (Database::getConnectionInfo($connection_key)) {
      return Database::getConnection('default', $connection_key);
    }

    // Ensure the database directory exists.
    $this->ensureDatabaseDirectory($index_id);

    // Add the connection info.
    $db_path = $this->getDatabasePath($index_id);
    Database::addConnectionInfo($connection_key, 'default', [
      'driver' => 'sqlite',
      'database' => $db_path,
    ]);

    $connection = Database::getConnection('default', $connection_key);

    // Apply performance PRAGMAs via PDO.
    $this->applyPragmas($index_id);

    return $connection;
  }

  /**
   * {@inheritdoc}
   */
  public function closeConnection(string $index_id): void {
    $connection_key = $this->getConnectionKey($index_id);

    // Remove Drupal connection.
    if (Database::getConnectionInfo($connection_key)) {
      Database::closeConnection('default', $connection_key);
      Database::removeConnection($connection_key);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getDatabasePath(string $index_id): string {
    // Check for custom path first.
    if (isset($this->customPaths[$index_id])) {
      $base_path = $this->customPaths[$index_id];
    }
    else {
      $config = $this->configFactory->get('search_api_sqlite.settings');
      $base_path = $config->get('database_path') ?? 'private://search_api_sqlite';
    }

    $real_path = $this->fileSystem->realpath($base_path);
    if ($real_path) {
      return $real_path . '/' . $index_id . '.sqlite';
    }

    // Fall back to constructing the path manually for private://.
    if (str_starts_with($base_path, 'private://')) {
      $private_path = $this->fileSystem->realpath('private://');
      if (!$private_path) {
        throw new \RuntimeException('Private file system is not configured.');
      }

      $relative = substr($base_path, strlen('private://'));
      return $private_path . '/' . $relative . '/' . $index_id . '.sqlite';
    }

    throw new \RuntimeException(sprintf('Cannot resolve database path: %s', $base_path));
  }

  /**
   * {@inheritdoc}
   */
  public function databaseExists(string $index_id): bool {
    $path = $this->getDatabasePath($index_id);
    return file_exists($path);
  }

  /**
   * {@inheritdoc}
   */
  public function deleteDatabase(string $index_id): bool {
    // Close connections first.
    $this->closeConnection($index_id);

    $path = $this->getDatabasePath($index_id);

    $deleted = TRUE;

    // Delete main database file.
    if (file_exists($path)) {
      $deleted = unlink($path);
    }

    // Delete WAL file if it exists.
    $wal_path = $path . '-wal';
    if (file_exists($wal_path)) {
      unlink($wal_path);
    }

    // Delete SHM file if it exists.
    $shm_path = $path . '-shm';
    if (file_exists($shm_path)) {
      unlink($shm_path);
    }

    if ($deleted) {
      $this->logger->info('Deleted SQLite database for index @index', [
        '@index' => $index_id,
      ]);
    }

    return $deleted;
  }

  /**
   * {@inheritdoc}
   */
  public function getPdo(string $index_id): \PDO {
    $connection = $this->getConnection($index_id);

    // Drupal's SQLite driver provides access to the underlying PDO connection.
    // This is needed for FTS5 special commands that use non-standard SQL.
    $pdo = $connection->getClientConnection();
    if (!$pdo instanceof \PDO) {
      throw new \RuntimeException('Could not get PDO connection from Drupal database wrapper.');
    }

    return $pdo;
  }

  /**
   * {@inheritdoc}
   */
  public function createTempItemsTable(string $index_id, array $item_ids, string $purpose = 'items'): string {
    $connection = $this->getConnection($index_id);

    // Use index_id and purpose in table name to avoid conflicts.
    $safe_index = preg_replace('/[^a-z0-9_]/', '_', $index_id);
    $safe_purpose = preg_replace('/[^a-z0-9_]/', '_', $purpose);
    $temp_table = 'temp_' . $safe_purpose . '_' . $safe_index;

    // Create temp table using raw query (CREATE TEMP TABLE not supported by
    // Schema API). SQLite temp tables live in the 'temp' schema.
    $connection->query(sprintf(
      'CREATE TEMP TABLE IF NOT EXISTS %s (item_id TEXT PRIMARY KEY)',
      $temp_table
    ));

    // Clear any existing data (in case of reuse within same connection).
    $connection->delete($temp_table)->execute();

    // Batch insert item IDs using Drupal DB API.
    // De-duplicate first to avoid PRIMARY KEY conflicts.
    $unique_ids = array_unique($item_ids);
    $batches = array_chunk($unique_ids, self::INSERT_BATCH_SIZE);

    foreach ($batches as $batch) {
      $insert = $connection->insert($temp_table)->fields(['item_id']);
      foreach ($batch as $item_id) {
        $insert->values(['item_id' => $item_id]);
      }

      $insert->execute();
    }

    return $temp_table;
  }

  /**
   * {@inheritdoc}
   */
  public function dropTempTable(string $index_id, string $table_name): void {
    $connection = $this->getConnection($index_id);
    // Use query() for DROP TABLE as Schema API doesn't handle temp tables.
    $connection->query(sprintf('DROP TABLE IF EXISTS %s', $table_name));
  }

  /**
   * {@inheritdoc}
   */
  public function shouldUseTempTable(int $count): bool {
    return $count >= self::TEMP_TABLE_THRESHOLD;
  }

  /**
   * {@inheritdoc}
   */
  public function getTempTableThreshold(): int {
    return self::TEMP_TABLE_THRESHOLD;
  }

  /**
   * Gets the connection key for an index.
   *
   * @param string $index_id
   *   The Search API index ID.
   *
   * @return string
   *   The connection key.
   */
  private function getConnectionKey(string $index_id): string {
    return self::CONNECTION_PREFIX . $index_id;
  }

  /**
   * Ensures the database directory exists.
   *
   * @param string $index_id
   *   The index ID to get the directory for.
   *
   * @throws \RuntimeException
   *   If the directory cannot be created.
   */
  private function ensureDatabaseDirectory(string $index_id): void {
    // Get the base path (custom or default).
    if (isset($this->customPaths[$index_id])) {
      $base_path = $this->customPaths[$index_id];
    }
    else {
      $config = $this->configFactory->get('search_api_sqlite.settings');
      $base_path = $config->get('database_path') ?? 'private://search_api_sqlite';
    }

    $real_path = $this->fileSystem->realpath($base_path);
    if ($real_path && is_dir($real_path)) {
      return;
    }

    // Handle stream wrappers.
    if (str_starts_with($base_path, 'private://')) {
      $private_path = $this->fileSystem->realpath('private://');
      if (!$private_path) {
        throw new \RuntimeException('Private file system is not configured. Please configure the private file path in settings.php.');
      }

      $relative = substr($base_path, strlen('private://'));
      $directory = $private_path . '/' . $relative;
    }
    elseif (str_starts_with($base_path, 'public://')) {
      $public_path = $this->fileSystem->realpath('public://');
      if (!$public_path) {
        throw new \RuntimeException('Public file system is not configured.');
      }

      $relative = substr($base_path, strlen('public://'));
      $directory = $public_path . '/' . $relative;
    }
    else {
      // Assume it's an absolute path.
      $directory = $base_path;
    }

    if (!is_dir($directory) && (!mkdir($directory, 0775, TRUE) && !is_dir($directory))) {
      throw new \RuntimeException(sprintf('Failed to create database directory: %s', $directory));
    }
  }

  /**
   * Applies performance PRAGMAs to a connection.
   *
   * @param string $index_id
   *   The Search API index ID.
   */
  private function applyPragmas(string $index_id): void {
    $connection_key = $this->getConnectionKey($index_id);
    $connection = Database::getConnection('default', $connection_key);

    // Use query() for PRAGMA statements as they don't work with prepared
    // statements in some contexts.
    // Enable WAL mode for better concurrent read/write performance.
    $connection->query('PRAGMA journal_mode = WAL');
    // NORMAL is safe with WAL - data survives OS crash, not just app crash.
    $connection->query('PRAGMA synchronous = NORMAL');
    // 64MB cache per connection (-N = N KiB). Balances memory vs performance.
    $connection->query('PRAGMA cache_size = -65536');
    // Keep temp tables and indices in memory for faster sorts/joins.
    $connection->query('PRAGMA temp_store = MEMORY');
    // Memory-map up to 256MB for faster reads (OS handles paging).
    $connection->query('PRAGMA mmap_size = 268435456');
    // Wait up to 5 seconds for locks instead of failing immediately.
    $connection->query('PRAGMA busy_timeout = 5000');
  }

}
