<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Database;

/**
 * Defines the interface for FTS5-specific operations.
 *
 * Main search queries use Drupal Database API with ->where() for MATCH.
 * This service handles FTS5 maintenance operations like optimize and rebuild.
 */
interface Fts5QueryRunnerInterface {

  /**
   * Optimizes the FTS5 index.
   *
   * @param string $index_id
   *   The Search API index ID.
   * @param string $table_name
   *   The FTS5 table name.
   */
  public function optimize(string $index_id, string $table_name): void;

  /**
   * Rebuilds the FTS5 index.
   *
   * @param string $index_id
   *   The Search API index ID.
   * @param string $table_name
   *   The FTS5 table name.
   */
  public function rebuild(string $index_id, string $table_name): void;

  /**
   * Gets the total count of documents in the FTS5 index.
   *
   * @param string $index_id
   *   The Search API index ID.
   * @param string $table_name
   *   The FTS5 table name.
   *
   * @return int
   *   The document count.
   */
  public function getDocumentCount(string $index_id, string $table_name): int;

  /**
   * Gets the count of documents matching an FTS5 query.
   *
   * @param string $index_id
   *   The Search API index ID.
   * @param string $table_name
   *   The FTS5 table name.
   * @param string $match_query
   *   The FTS5 MATCH query string.
   *
   * @return int
   *   The matching document count.
   */
  public function getMatchCount(string $index_id, string $table_name, string $match_query): int;

  /**
   * Checks if any documents match an FTS5 query.
   *
   * More efficient than getMatchCount() when you only need to know if matches
   * exist, as it stops at the first match.
   *
   * @param string $index_id
   *   The Search API index ID.
   * @param string $table_name
   *   The FTS5 table name.
   * @param string $match_query
   *   The FTS5 MATCH query string.
   *
   * @return bool
   *   TRUE if at least one document matches, FALSE otherwise.
   */
  public function matchExists(string $index_id, string $table_name, string $match_query): bool;

}
