<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Search;

use Drupal\Core\Database\Query\SelectInterface;
use Drupal\search_api_sqlite\Database\SchemaManagerInterface;
use Drupal\search_api_sqlite\Index\FieldTypeMapperInterface;

/**
 * Helper for building condition subqueries.
 *
 * Provides shared functionality for QueryBuilder and FacetBuilder
 * to avoid code duplication.
 */
final readonly class ConditionHelper implements ConditionHelperInterface {

  /**
   * Constructs a ConditionHelper instance.
   *
   * @param \Drupal\search_api_sqlite\Database\SchemaManagerInterface $schemaManager
   *   The schema manager.
   * @param \Drupal\search_api_sqlite\Index\FieldTypeMapperInterface $fieldTypeMapper
   *   The field type mapper.
   */
  public function __construct(
    private SchemaManagerInterface $schemaManager,
    private FieldTypeMapperInterface $fieldTypeMapper,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function getValueColumnForField(string $field_name, array $fields, mixed $value): string {
    // Try to get field type from index definition.
    if (isset($fields[$field_name])) {
      $field = $fields[$field_name];
      $type = $field->getType();
      $storage_type = $this->fieldTypeMapper->getStorageType($type);
      return $this->fieldTypeMapper->getFieldDataColumn($storage_type);
    }

    // Fallback to value type detection.
    return $this->getValueColumn($value);
  }

  /**
   * {@inheritdoc}
   */
  public function getValueColumn(mixed $value): string {
    // Handle arrays - check first element.
    if (is_array($value)) {
      $value = reset($value);
    }

    if (is_int($value) || is_bool($value)) {
      return 'value_integer';
    }

    if (is_float($value)) {
      return 'value_decimal';
    }

    return 'value_string';
  }

  /**
   * {@inheritdoc}
   */
  public function buildLanguageSubquery(
    $connection,
    string $index_id,
    mixed $value,
    string $operator,
  ): ?SelectInterface {
    $items_table = $this->schemaManager->getItemsTableName($index_id);
    $subquery = $connection->select($items_table, 'it');
    $subquery->fields('it', ['item_id']);

    switch (strtoupper($operator)) {
      case 'IN':
        $subquery->condition('it.language', $value, 'IN');
        break;

      case 'NOT IN':
        $subquery->condition('it.language', $value, 'NOT IN');
        break;

      case '=':
        $subquery->condition('it.language', $value);
        break;

      case '<>':
      case '!=':
        $subquery->condition('it.language', $value, '<>');
        break;

      default:
        return NULL;
    }

    return $subquery;
  }

  /**
   * {@inheritdoc}
   */
  public function buildLanguageCondition(
    $connection,
    string $index_id,
    mixed $value,
    string $operator,
  ): array {
    $items_table = $this->schemaManager->getItemsTableName($index_id);
    $subquery = $connection->select($items_table, 'it');
    $subquery->fields('it', ['item_id']);

    $outer_operator = 'IN';
    switch (strtoupper($operator)) {
      case 'IN':
        $subquery->condition('it.language', $value, 'IN');
        break;

      case 'NOT IN':
        $subquery->condition('it.language', $value, 'IN');
        $outer_operator = 'NOT IN';
        break;

      case '=':
        $subquery->condition('it.language', $value);
        break;

      case '<>':
      case '!=':
        $subquery->condition('it.language', $value);
        $outer_operator = 'NOT IN';
        break;

      default:
        return [NULL, 'IN'];
    }

    return [$subquery, $outer_operator];
  }

}
