<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Spellcheck;

/**
 * Interface for spell check handler.
 */
interface SpellCheckHandlerInterface {

  /**
   * Checks if spell checking is available.
   *
   * @return bool
   *   TRUE if php-spellchecker library is installed.
   */
  public function isAvailable(): bool;

  /**
   * Checks search query for misspellings.
   *
   * @param string $query
   *   The search query.
   * @param string $backend
   *   Spellchecker backend (aspell, hunspell, pspell).
   * @param string $language
   *   Language code.
   * @param array<string, mixed> $options
   *   Backend-specific options:
   *   - binary_path: Path to aspell/hunspell binary (default: auto-detect).
   *
   * @return array{has_misspellings: bool, collation: string, suggestions: array<string, array<string>>}
   *   Result array:
   *   - has_misspellings: bool indicating if misspellings were found
   *   - collation: the best corrected full query string
   *   - suggestions: map of misspelled word to array of alternatives
   */
  public function checkQuery(string $query, string $backend, string $language, array $options = []): array;

  /**
   * Gets available spellchecker backends.
   *
   * @return array<string>
   *   Available backend names.
   */
  public function getAvailableBackends(): array;

}
