<?php

declare(strict_types=1);

namespace Drupal\Tests\search_api_sqlite\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\search_api\Entity\Index;
use Drupal\search_api\Entity\Server;
use Drupal\search_api\Utility\Utility;
use Drupal\search_api_sqlite\Plugin\search_api\backend\SqliteFts5;
use Drupal\Tests\search_api\Functional\ExampleContentTrait;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests FTS5 native highlighting via the processor.
 *
 * @group search_api_sqlite
 */
#[CoversClass(SqliteFts5::class)]
#[Group('search_api_sqlite')]
class HighlightTest extends KernelTestBase {

  use SqliteDatabaseCleanupTrait;
  use ExampleContentTrait;

  /**
   * {@inheritdoc}
   *
   * @var list<string>
   */
  protected static $modules = [
    'entity_test',
    'field',
    'system',
    'filter',
    'text',
    'user',
    'search_api',
    'search_api_sqlite',
    'search_api_test',
    'search_api_test_sqlite',
    'search_api_test_example_content',
  ];

  /**
   * A search server ID.
   */
  protected string $serverId = 'sapi_sqlite_search_server';

  /**
   * A search index ID.
   */
  protected string $indexId = 'sapi_sqlite_search_index';

  /**
   * {@inheritdoc}
   */
  public function setUp(): void {
    parent::setUp();

    $this->installSchema('search_api', ['search_api_item']);
    $this->installSchema('user', ['users_data']);
    $this->installEntitySchema('entity_test_mulrev_changed');
    $this->installEntitySchema('search_api_task');
    $this->installConfig('search_api');

    // Do not use a batch for tracking the initial items after creating an
    // index when running the tests via the GUI.
    if (!Utility::isRunningInCli()) {
      \Drupal::state()->set('search_api_use_tracking_batch', FALSE);
    }

    $this->installConfig(['search_api_test_example_content']);

    // Install server and index configs with unique database path.
    $this->installSqliteTestConfig(
      'search_api.server.sapi_sqlite_search_server',
      'search_api.index.sapi_sqlite_search_index',
      'search_api_test_sqlite',
    );

    $this->setUpExampleStructure();
    $this->insertExampleContent();

    $this->indexItems($this->indexId);
  }

  /**
   * Tests that native FTS5 highlighting works via query option.
   */
  public function testNativeHighlighting(): void {
    $index = Index::load($this->indexId);
    $this->assertNotNull($index);

    $server = Server::load($this->serverId);
    $this->assertNotNull($server);

    $backend = $server->getBackend();
    $this->assertInstanceOf(SqliteFts5::class, $backend);

    // Create a query with highlighting option set.
    $query = $index->query();
    $query->keys('foo');
    $query->setOption('search_api_sqlite_highlight', [
      'prefix' => '<mark>',
      'suffix' => '</mark>',
      'excerpt_length' => 64,
      'exclude_fields' => [],
    ]);

    $results = $query->execute();

    // "foo" appears in several test items.
    $this->assertGreaterThan(0, $results->getResultCount());

    // Check that at least one result has an excerpt with highlighting.
    $found_highlight = FALSE;
    foreach ($results->getResultItems() as $item) {
      $excerpt = $item->getExcerpt();
      if ($excerpt !== NULL && str_contains($excerpt, '<mark>')) {
        $found_highlight = TRUE;
        // Verify the highlight structure.
        $this->assertStringContainsString('<mark>', $excerpt);
        $this->assertStringContainsString('</mark>', $excerpt);
        break;
      }
    }

    $this->assertTrue($found_highlight, 'At least one result has a highlighted excerpt');
  }

  /**
   * Tests highlighting with excluded fields.
   */
  public function testHighlightingWithExcludedFields(): void {
    $index = Index::load($this->indexId);
    $this->assertNotNull($index);

    // Create a query excluding the body field.
    $query = $index->query();
    $query->keys('test');
    $query->setOption('search_api_sqlite_highlight', [
      'prefix' => '<em>',
      'suffix' => '</em>',
      'excerpt_length' => 32,
      'exclude_fields' => ['body'],
    ]);

    $results = $query->execute();

    // Verify search works with excluded field.
    $this->assertGreaterThan(0, $results->getResultCount());
  }

  /**
   * Tests that highlighting is skipped without search keys.
   */
  public function testNoHighlightingWithoutKeys(): void {
    $index = Index::load($this->indexId);
    $this->assertNotNull($index);

    // Create a query without search keys.
    $query = $index->query();
    $query->setOption('search_api_sqlite_highlight', [
      'prefix' => '<mark>',
      'suffix' => '</mark>',
      'excerpt_length' => 64,
      'exclude_fields' => [],
    ]);

    $results = $query->execute();

    // Should return results but no highlights.
    $this->assertGreaterThan(0, $results->getResultCount());

    // No result should have a highlighted excerpt.
    foreach ($results->getResultItems() as $item) {
      $excerpt = $item->getExcerpt();
      if ($excerpt !== NULL) {
        $this->assertStringNotContainsString(
          '<mark>',
          $excerpt,
          'No highlighting when no keys'
        );
      }
    }
  }

}
