<?php

namespace Drupal\search_api_term_with_depth\Hook;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\search_api\SearchApiException;
use Drupal\search_api_term_with_depth\Plugin\views\filter\SearchApiTermWithDepth as TermWithDepthFilter;
use Drupal\search_api_term_with_depth\Plugin\views\argument\SearchApiTermWithDepth as TermWithDepthArgument;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\Attribute\Autowire;

/**
 * Implements hooks for the Search API Term with Depth module.
 */
final class SearchApiTermWithDepthHooks {

  /**
   * Constructs a new SearchApiTermWithDepthHooks object.
   *
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger for the 'search_api_term_with_depth' channel.
   */
  public function __construct(
    private readonly ModuleHandlerInterface $moduleHandler,
    private readonly EntityTypeManagerInterface $entityTypeManager,
    #[Autowire(service: 'logger.channel.search_api_term_with_depth')]
    private readonly LoggerInterface $logger,
  ) {}

  /**
   * Implements hook_views_plugins_filter_alter().
   *
   * Dynamically adds the term with depth filter plugin.
   *
   * @param array $plugins
   *   The array of Views filter plugins, passed by reference.
   *
   * @phpstan-ignore-next-line
   */
  #[Hook('views_plugins_filter_alter')]
  public function viewsPluginsFilterAlter(array &$plugins): void {
    // We have to include the term filter handler like this, since adding it
    // directly (i.e., with an annotation) would cause fatal errors on sites
    // without the Taxonomy or Search API modules.
    if ($this->moduleHandler->moduleExists('taxonomy') && $this->moduleHandler->moduleExists('search_api')) {
      $plugins['search_api_term_with_depth'] = [
        'plugin_type' => 'filter',
        'id' => 'search_api_term_with_depth',
        'class' => TermWithDepthFilter::class,
        'provider' => 'search_api_term_with_depth',
      ];
    }
  }

  /**
   * Implements hook_views_plugins_argument_alter().
   */
  #[Hook('views_plugins_argument_alter')]
  public function viewsPluginsArgumentAlter(array &$plugins): void {
    if ($this->moduleHandler->moduleExists('taxonomy') && $this->moduleHandler->moduleExists('search_api')) {
      $plugins['search_api_term_with_depth'] = [
        'plugin_type' => 'argument',
        'id' => 'search_api_term_with_depth',
        'class' => TermWithDepthArgument::class,
        'provider' => 'search_api_term_with_depth',
      ];
    }
  }

  /**
   * Implements hook_views_data_alter().
   *
   * Replaces the default filter and argument for indexed taxonomy term fields
   * with our custom argument or filter plugin.
   *
   * @param array $data
   *   The array of Views data, passed by reference.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   *
   * @phpstan-ignore-next-line
   */
  #[Hook('views_data_alter')]
  public function viewsDataAlter(array &$data): void {
    $indexes = $this->entityTypeManager
      ->getStorage('search_api_index')
      ->loadMultiple();

    // Loop through each Search API index to find its corresponding Views data.
    foreach ($indexes as $index) {
      $index_id = $index->id();
      $views_table_key = 'search_api_index_' . $index_id;

      // Check if the Views data for this index exists.
      if (!isset($data[$views_table_key])) {
        continue;
      }

      try {
        // Get all fields for the index.
        $fields = $index->getFields();
        foreach ($fields as $field_id => $field) {
          // We only care about fields that are entity references to
          // taxonomy terms.
          $definition = $field->getDataDefinition();
          if ($definition->getSetting('target_type') === 'taxonomy_term') {
            // In Views data, field IDs can be altered if there are duplicates,
            // so we need to find the correct key in the $data array.
            // We can find this by checking the 'real field' property.
            foreach ($data[$views_table_key] as $key => $field_info) {
              // Match fields corresponding to this taxonomy field.
              if (($field_info['real field'] ?? $key) === $field_id) {
                // Replace the filter plugin.
                if (isset($field_info['filter']['id'])) {
                  $data[$views_table_key][$key]['filter']['id'] = 'search_api_term_with_depth';
                }
                // Replace the argument plugin.
                if (isset($field_info['argument']['id'])) {
                  $data[$views_table_key][$key]['argument']['id'] = 'search_api_term_with_depth';
                }
              }
            }
          }
        }
      }
      catch (SearchApiException $e) {
        // Log any exceptions that occur while getting field data.
        $this->logger->error('Error altering Views data for index {index}: {message}', [
          'index' => $index_id,
          'message' => $e->getMessage(),
        ]);
      }
    }
  }

}
