<?php

namespace Drupal\search_api_term_with_depth\Plugin\views\argument;

use Drupal\Core\Form\FormStateInterface;
use Drupal\search_api\Plugin\views\argument\SearchApiTerm;
use Drupal\views\Attribute\ViewsArgument;

/**
 * Defines a contextual filter for searching taxonomy terms with depth.
 *
 * @ingroup views_argument_handlers
 */
#[ViewsArgument(id: "search_api_term_with_depth")]
class SearchApiTermWithDepth extends SearchApiTerm {

  /**
   * {@inheritdoc}
   */
  public function defineOptions(): array {
    $options = parent::defineOptions();
    $options['depth'] = ['default' => 0];
    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state): void {
    parent::buildOptionsForm($form, $form_state);

    $form['depth'] = [
      '#type' => 'weight',
      '#title' => $this->t('Depth'),
      '#default_value' => $this->options['depth'],
      '#description' => $this->t('The depth will match content tagged with terms in the hierarchy. For example, if you have the term "fruit" and a child term "apple", with a depth of 1 (or higher) then filtering for the term "fruit" will get content that are tagged with "apple" as well as "fruit". If negative, the reverse is true; searching for "apple" will also pick up content tagged with "fruit" if depth is -1 (or lower).'),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function query($group_by = FALSE): void {
    // Parse the URL argument into TIDs ($this->value).
    $this->fillValue();

    // Ensure initial values are integers to match storage IDs.
    if (!empty($this->value)) {
      $this->value = array_map('intval', $this->value);
    }

    // Expand the TIDs if depth is configured.
    if (!empty($this->value) && !empty($this->options['depth'])) {
      $depth = (int) $this->options['depth'];
      $expanded_tids = $this->value;

      foreach ($this->value as $tid) {
        if ($depth > 0) {
          $children = $this->getChildrenTids($tid);
          array_push($expanded_tids, ...$children);
        }
        elseif ($depth < 0) {
          $parents = $this->getParentsTids($tid);
          array_push($expanded_tids, ...$parents);
        }
      }

      // De-duplicate the results.
      $this->value = array_values(array_unique($expanded_tids));
    }

    // Apply the condition directly to the query.
    if (!empty($this->value)) {
      // We force the use of IN (or NOT IN) because we have potentially expanded
      // a single term into a list of terms (parent + children).
      $operator = !empty($this->options['not']) ? 'NOT IN' : 'IN';
      $this->query->addCondition($this->realField, $this->value, $operator);
    }
  }

  /**
   * Gets the child term IDs for a given parent term ID.
   *
   * @param string|int $parent_tid
   *   The parent taxonomy term ID.
   *
   * @return array
   *   An array of child taxonomy term IDs.
   */
  private function getChildrenTids(string|int $parent_tid): array {
    $children = $this->getTermStorage()->loadChildren($parent_tid);
    return $children ? array_map('intval', array_keys($children)) : [];
  }

  /**
   * Gets the parent term IDs for a given child term ID.
   *
   * @param string|int $child_tid
   *   The child taxonomy term ID.
   *
   * @return array
   *   An array of parent taxonomy term IDs.
   */
  private function getParentsTids(string|int $child_tid): array {
    $parents = $this->getTermStorage()->loadParents($child_tid);
    return $parents ? array_map('intval', array_keys($parents)) : [];
  }

}
