<?php

namespace Drupal\search_api_term_with_depth\Plugin\views\filter;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\search_api\Plugin\views\filter\SearchApiTerm;
use Drupal\views\Attribute\ViewsFilter;

/**
 * Provides a Views filter for searching taxonomy terms with depth.
 *
 * This handler uses the taxonomy storage to find parent or child terms to
 * include in the filter conditions.
 *
 * @ingroup views_field_handlers
 */
#[ViewsFilter(id: "search_api_term_with_depth")]
class SearchApiTermWithDepth extends SearchApiTerm implements ContainerFactoryPluginInterface {

  /**
   * {@inheritdoc}
   */
  public function operatorOptions($which = 'title'): array {
    $op = [
      'or' => $this->t('Is one of'),
    ];
    if ($this->options['depth'] == 0) {
      $op += [
        'and' => $this->t('Is all of'),
        'not' => $this->t('Is none of'),
      ];
      // If the definition allows for the empty operator, add it.
      if (!empty($this->definition['allow empty'])) {
        $op += [
          'empty' => $this->t('Is empty (NULL)'),
          'not empty' => $this->t('Is not empty (NOT NULL)'),
        ];
      }
    }

    return $op;
  }

  /**
   * {@inheritdoc}
   */
  protected function defineOptions(): array {
    $options = parent::defineOptions();
    $options['depth'] = ['default' => 0];
    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildExtraOptionsForm(&$form, FormStateInterface $form_state): void {
    parent::buildExtraOptionsForm($form, $form_state);

    $form['depth'] = [
      '#type' => 'weight',
      '#title' => $this->t('Depth'),
      '#default_value' => $this->options['depth'],
      '#description' => $this->t('The depth will match content tagged with terms in the hierarchy. For example, if you have the term "fruit" and a child term "apple", with a depth of 1 (or higher) then filtering for the term "fruit" will get content that are tagged with "apple" as well as "fruit". If negative, the reverse is true; searching for "apple" will also pick up content tagged with "fruit" if depth is -1 (or lower).'),
    ];
  }

  /**
   * {@inheritdoc}
   */
  protected function opHelper(): void {
    $this->value = array_filter($this->value, [static::class, 'arrayFilterZero']);
    if (empty($this->value)) {
      return;
    }

    if ($this->operator !== 'and') {
      $operator = $this->operator === 'not' ? 'NOT IN' : 'IN';
      $selections = $this->value;
      $selection_count = is_countable($selections) ? count($selections) : 0;
      if ($selection_count === 0) {
        return;
      }
      foreach ($selections as $selection) {
        if ($this->options['depth'] > 0) {
          $children = $this->getChildrenTids($selection);
          array_push($selections, ...$children);
        }
        elseif ($this->options['depth'] < 0) {
          $parents = $this->getParentsTids($selection);
          array_push($selections, ...$parents);
        }
      }
      $this->getQuery()
        ->addCondition($this->realField, $selections, $operator, $this->options['group']);
      return;
    }

    $condition_group = $this->getQuery()->createConditionGroup();
    foreach ($this->value as $value) {
      $condition_group->addCondition($this->realField, $value, '=');
    }
    $this->getQuery()
      ->addConditionGroup($condition_group, $this->options['group']);
  }

  /**
   * Gets the child term IDs for a given parent term ID.
   *
   * @param string|int $parent_tid
   *   The parent taxonomy term ID.
   *
   * @return array
   *   An array of child taxonomy term IDs.
   */
  private function getChildrenTids(string|int $parent_tid): array {
    $children = $this->termStorage->loadChildren($parent_tid);
    return $children ? array_keys($children) : [];
  }

  /**
   * Gets the parent term IDs for a given child term ID.
   *
   * @param string|int $child_tid
   *   The child taxonomy term ID.
   *
   * @return array
   *   An array of parent taxonomy term IDs.
   */
  private function getParentsTids(string|int $child_tid): array {
    $parents = $this->termStorage->loadParents($child_tid);
    return $parents ? array_keys($parents) : [];
  }

}
