((Drupal, TypesenseInstantSearchAdapter, instantsearch) => {
  Drupal.behaviors.search = {
    attach(context, settings) {
      const [searchbox] = once('searchbox', '#searchbox', context);
      if (searchbox === undefined) {
        return;
      }

      const typesenseInstantsearchAdapter = new TypesenseInstantSearchAdapter({
        server: settings.search_api_typesense.server,
        // Search parameters that are common to all collections/indices go here.
        additionalSearchParameters: {
          exclude_fields: 'embedding',
          exhaustive_search: true,
        },
        // Search parameters that need to be *overridden* on a
        // per-collection-basis go here. This is useful for querying in
        // different collections simultaneously.
        // Look at the documentation for more details:
        // https://github.com/typesense/typesense-instantsearch-adapter/tree/v2.10.0-0?tab=readme-ov-file#index
        collectionSpecificSearchParameters:
          settings.search_api_typesense.collection_specific_search_parameters,
      });
      const searchClient = typesenseInstantsearchAdapter.searchClient;

      const collections =
        settings.search_api_typesense.collection_render_parameters;
      const firstCollection = collections[0];
      const search = instantsearch({
        searchClient,
        // The default index is required, and in a Federated/Multi-Index Search
        // it can be the name of the first index; then the Widgets inside
        // Index components will override this.
        indexName: firstCollection.collection_name,
      });

      const getCollectionWidgets = (collection) => [
        instantsearch.widgets.stats({
          container: `#stats-${collection.collection_name}`,
        }),
        instantsearch.widgets.infiniteHits({
          container: `#hits-${collection.collection_name}`,
          cssClasses: {
            list: 'search-results',
            loadMore: 'btn btn--load-more',
          },
          templates: {
            item(hit, { html, components }) {
              if (settings.search_api_typesense.debug) {
                return html`
                  <article class="search-result-debug">
                    <ul>
                      ${Object.keys(hit).map(
                        (key) =>
                          html`<li>
                            <strong>${key}:</strong> ${JSON.stringify(
                              hit[key],
                              null,
                              2,
                            )}
                          </li>`,
                      )}
                    </ul>
                  </article>
                `;
              }
              if (hit.rendered_item) {
                return html`${hit.rendered_item}`;
              }
              return html`
                <article>
                  <h3>${components.Highlight({ hit, attribute: 'title' })}</h3>
                  <small> (${hit.id})</small>
                </article>
              `;
            },
          },
        }),
      ];

      search.addWidgets([
        instantsearch.widgets.configure({
          hitsPerPage: settings.search_api_typesense.hits_per_page,
          // The facetFilters are used to filter the results based on the
          // current language code. This is useful for multilingual sites.
          ...(settings.search_api_typesense.current_langcode && {
            facetFilters: [
              `langcode:${settings.search_api_typesense.current_langcode}`,
            ],
          }),
        }),
        instantsearch.widgets.searchBox({
          container: '#searchbox',
          placeholder: 'Search...',
        }),
        ...getCollectionWidgets(firstCollection),
      ]);

      if (collections.length > 1) {
        collections.slice(1).forEach((collection) => {
          search.addWidgets([
            instantsearch.widgets
              .index({ indexName: collection.collection_name })
              .addWidgets(getCollectionWidgets(collection)),
          ]);
        });
      }

      const facetWidgets = {
        string: (facet) =>
          instantsearch.widgets.refinementList({
            container: `#facet-${facet.name}`,
            attribute: facet.name,
            searchable: true,
          }),
        number: (facet) =>
          instantsearch.widgets.rangeSlider({
            container: `#facet-${facet.name}`,
            attribute: facet.name,
          }),
        bool: (facet) =>
          instantsearch.widgets.toggleRefinement({
            container: `#facet-${facet.name}`,
            attribute: facet.name,
            templates: {
              labelText: facet.label,
            },
          }),
      };

      settings.search_api_typesense.facets.forEach((facet) => {
        if (facetWidgets[facet.type]) {
          search.addWidgets([facetWidgets[facet.type](facet)]);
        }
      });

      search.start();
    },
  };
})(Drupal, TypesenseInstantSearchAdapter, instantsearch); // eslint-disable-line
