<?php

namespace Drupal\search_api_vragen_ai;

use Drupal\search_api_vragen_ai\Client\DocumentItem;
use Drupal\search_api_vragen_ai\Client\SystemItem;
use Psr\Http\Client\ClientInterface as HttpClientInterface;
use Swis\JsonApi\Client\Client as JsonApiClient;
use Swis\JsonApi\Client\DocumentClient;
use Swis\JsonApi\Client\Interfaces\ClientInterface;
use Swis\JsonApi\Client\Interfaces\ResponseParserInterface;
use Swis\JsonApi\Client\Interfaces\TypeMapperInterface;
use Swis\JsonApi\Client\Parsers\ResponseParser;
use Swis\JsonApi\Client\TypeMapper;

/**
 * Client to interact with the Vragen.ai API.
 */
class Client extends DocumentClient {

  /**
   * The client.
   */
  private ClientInterface $client;

  /**
   * Constructor.
   *
   * @param \Swis\JsonApi\Client\Interfaces\ClientInterface $client
   *   The client.
   * @param \Swis\JsonApi\Client\Interfaces\ResponseParserInterface $parser
   *   The response parser.
   * @param string|null $apiEndpoint
   *   The Vragen.ai API endpoint.
   * @param string|null $apiToken
   *   The Vragen.ai API token.
   */
  public function __construct(ClientInterface $client, ResponseParserInterface $parser, ?string $apiEndpoint = NULL, ?string $apiToken = NULL) {
    parent::__construct($client, $parser);
    $this->client = $client;

    $this->setBaseUri($apiEndpoint);

    /** @var \Swis\JsonApi\Client\Client $baseClient */
    $baseClient = $this->client;
    $baseClient->setDefaultHeaders(array_merge(
      $baseClient->getDefaultHeaders(), [
        'Authorization' => sprintf('Bearer %s', $apiToken),
      ]
    ));
  }

  /**
   * Create a new instance of the client.
   *
   * @param \Swis\JsonApi\Client\Interfaces\TypeMapperInterface|null $typeMapper
   *   The type mapper.
   * @param \Psr\Http\Client\ClientInterface|null $client
   *   The HTTP client.
   * @param string|null $apiEndpoint
   *   The Vragen.ai API endpoint.
   * @param string|null $apiToken
   *   The Vragen.ai API token.
   *
   * @return \Swis\JsonApi\Client\DocumentClient
   *   The client.
   */
  public static function create(?TypeMapperInterface $typeMapper = NULL, ?HttpClientInterface $client = NULL, ?string $apiEndpoint = NULL, ?string $apiToken = NULL): DocumentClient {
    return new self(
      client: new JsonApiClient($client),
      parser: ResponseParser::create($typeMapper),
      apiEndpoint: $apiEndpoint,
      apiToken: $apiToken,
    );
  }

  /**
   * Get the client.
   *
   * @return \Swis\JsonApi\Client\Interfaces\ClientInterface
   *   The client.
   */
  public function getClient(): ClientInterface {
    return $this->client;
  }

  /**
   * Get the type mapper.
   *
   * @return \Swis\JsonApi\Client\Interfaces\TypeMapperInterface
   *   The type mapper.
   */
  public static function getTypeMapper(): TypeMapperInterface {
    $typeMapper = new TypeMapper();
    $typeMapper->setMapping('documents', DocumentItem::class);
    $typeMapper->setMapping('systems', SystemItem::class);

    return $typeMapper;
  }

}
