<?php

namespace Drupal\search_api_vragen_ai\Plugin\search_api\backend;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\PluginFormInterface;
use Drupal\Core\Render\Element;
use Drupal\file\Entity\File;
use Drupal\media\MediaInterface;
use Drupal\search_api\Backend\BackendPluginBase;
use Drupal\search_api\IndexInterface;
use Drupal\search_api\Item\ItemInterface;
use Drupal\search_api\Plugin\PluginFormTrait;
use Drupal\search_api\Query\QueryInterface;
use Drupal\search_api_vragen_ai\Client;
use Drupal\search_api_vragen_ai\Client\DocumentRepository;
use Drupal\user\EntityOwnerInterface;
use Http\Discovery\Psr17FactoryDiscovery;
use Http\Discovery\Psr18ClientDiscovery;
use Psr\Http\Client\ClientInterface;
use Swis\JsonApi\Client\DocumentFactory;
use Swis\JsonApi\Client\Exceptions\ValidationException;
use Swis\JsonApi\Client\Interfaces\CollectionDocumentInterface;
use Swis\JsonApi\Client\Interfaces\ItemInterface as JsonApiItemInterface;
use Swis\JsonApi\Client\ItemHydrator;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * The Vragen.ai search backend.
 *
 * @SearchApiBackend(
 *   id = "vragen_ai",
 *   label = @Translation("Vragen.ai"),
 *   description = @Translation("Vragen.ai semantic search.")
 * )
 */
class VragenAiBackend extends BackendPluginBase implements PluginFormInterface {

  use PluginFormTrait;

  /**
   * The document repository.
   */
  protected DocumentRepository $documentRepository;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The HTTP client.
   *
   * @var \Psr\Http\Client\ClientInterface
   */
  protected ClientInterface $httpClient;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);

    $configFactory = $container->get('config.factory');

    $instance
      ->withConfigFactory($configFactory)
      ->withHttpClient(Psr18ClientDiscovery::find())
      ->withDocumentRepository(self::buildDefaultDocumentRepository($configFactory));

    return $instance;
  }

  /**
   * Set HTTP Client to use.
   *
   * @param \Psr\Http\Client\ClientInterface $client
   *   HTTP Client to use.
   *
   * @return $this
   *   The current instance.
   */
  public function withHttpClient(ClientInterface $client): self {
    $this->httpClient = $client;

    return $this;
  }

  /**
   * Set document repository to use.
   *
   * @param \Drupal\search_api_vragen_ai\Client\DocumentRepository $repository
   *   Document repository to use.
   *
   * @return $this
   *   The current instance.
   */
  public function withDocumentRepository(DocumentRepository $repository): self {
    $this->documentRepository = $repository;

    return $this;
  }

  /**
   * Set config factory.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   Config factory.
   *
   * @return $this
   *   The current instance.
   */
  public function withConfigFactory(ConfigFactoryInterface $configFactory): self {
    $this->configFactory = $configFactory;

    return $this;
  }

  /**
   * Document repository builder.
   *
   * @return \Drupal\search_api_vragen_ai\Client\DocumentRepository
   *   Document repository.
   */
  protected static function buildDefaultDocumentRepository(ConfigFactoryInterface $configFactory): DocumentRepository {
    $config = $configFactory->get('search_api_vragen_ai.settings');

    $apiEndpoint = $apiEndpoint ?? $config->get('vragen_ai_endpoint') ?? '';
    $apiToken = $apiToken ?? $config->get('vragen_ai_token') ?? '';

    $httpClient = Psr18ClientDiscovery::find();
    $client = Client::create(Client::getTypeMapper(), $httpClient, $apiEndpoint, $apiToken);

    return new DocumentRepository(
      $client,
      new DocumentFactory()
    );
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function indexItems(IndexInterface $index, array $items): array {
    $indexed = [];

    foreach ($items as $id => $item) {
      try {
        $this->indexItem($item);
        $indexed[] = $id;
      }
      catch (\Throwable $e) {
        $this->getLogger()->error($e->getMessage() . ' (Document: ' . $item->getId() . ')');
      }
    }

    return $indexed;
  }

  /**
   * Indexes the specified items.
   */
  protected function indexItem(ItemInterface $item): void {
    /** @var \Drupal\Core\Entity\Plugin\DataType\EntityAdapter $entity */
    $entity = $item->getOriginalObject();
    if (!$entity) {
      return;
    }

    $entity_value = $entity->getValue();
    $document = $this->firstOrCreateDocument($item->getId());

    $published_date = date('Y-m-d');
    if ($entity_value instanceof ContentEntityInterface && $entity_value->hasField('created')) {
      $published_date = date('Y-m-d', (int) $entity_value->get('created')->getString());
    }

    $datasource = $item->getDatasource();

    $meta_data = [
      'author' => $entity_value instanceof EntityOwnerInterface ? $entity_value->getOwner()->getDisplayName() : NULL,
      'entity_type' => $datasource->getEntityTypeId(),
      'bundle' => $datasource->getItemBundle($entity),
      'date' => $published_date,
    ];

    if ($entity_value instanceof MediaInterface) {
      $file_id = $entity_value->getSource()->getSourceFieldValue($entity_value);
      $file = File::load($file_id);

      if (!$file || $file->getMimeType() !== 'application/pdf') {
        return;
      }

      $content = NULL;
      $url = $file->createFileUrl(FALSE);
      $mime_type = $file->getMimeType();
    }
    else {
      $content = $this->getIndexContentForItem($item, $meta_data);
      $url = $datasource->getItemUrl($entity)?->setAbsolute()->toString();
      $mime_type = 'text/html';
    }

    $document->fill([
      'external_reference' => $item->getId(),
      'title' => $datasource->getItemLabel($entity) ?? 'Empty',
      'url' => $url,
      'mime_type' => $mime_type,
      'content' => $content,
      'meta_data' => $meta_data,
    ]);

    if ($document->isNew()) {
      $result = $this->documentRepository->create($document);
    }
    else {
      $result = $this->documentRepository->update($document);
    }

    if (!empty($result->getErrors()->isNotEmpty())) {
      $error = $result->getErrors()->first();
      throw new \Exception('Error when indexing item: ' . $error->getTitle() . ' (' . $error->getStatus() . ': ' . $error->getDetail() . ')');
    }
  }

  /**
   * {@inheritdoc}
   */
  public function deleteItems(IndexInterface $index, array $item_ids): void {
    foreach ($item_ids as $id) {
      $document = $this->firstOrCreateDocument($id);
      if ($document->isNew() || empty($document->getId())) {
        continue;
      }

      $this->documentRepository->delete($document->getId());
    }
  }

  /**
   * {@inheritdoc}
   */
  public function supportsDataType($type): bool {
    if ($type === 'vragen_ai_metadata') {
      return TRUE;
    }
    return parent::supportsDataType($type);
  }

  /**
   * {@inheritdoc}
   */
  public function deleteAllIndexItems(IndexInterface $index, $datasource_id = NULL): void {
    // Not supported.
  }

  /**
   * {@inheritdoc}
   */
  public function search(QueryInterface $query): void {
    $searchEndpoint = $this->configFactory->get('search_api_vragen_ai.settings')->get('vragen_ai_search_endpoint');
    if (filter_var($searchEndpoint, FILTER_VALIDATE_URL) === FALSE) {
      $this->getLogger()->warning('Vragen AI Search Endpoint is not set or invalid.');
      return;
    }

    $keys = $query->getKeys();
    if (empty($keys)) {
      return;
    }

    $searchPhrase = $this->getSearchPhrase($keys);
    $request = Psr17FactoryDiscovery::findRequestFactory()->createRequest('GET', $searchEndpoint . '&query=' . urlencode($searchPhrase));
    $request = $request->withHeader('Accept', 'application/json');

    $response = $this->httpClient->sendRequest($request);
    if ($response->getStatusCode() !== 200) {
      $this->getLogger()->error('Failed to search: ' . $response->getReasonPhrase());
      return;
    }

    $index = $query->getIndex();

    $body = json_decode($response->getBody()->getContents(), TRUE);
    if ($body === FALSE) {
      return;
    }

    $results = $query->getResults();
    if (isset($body['run_reference'])) {
      $results->setExtraData('search_api_run_reference', $body['run_reference']);
    }

    if (empty($body['results'])) {
      return;
    }

    $resultCount = 0;
    foreach ($body['results'] as $resultItem) {
      // We can safely skip items that don't have an external reference.
      // This happens when items are not indexed using Search API.
      if (!isset($resultItem['external_reference'])) {
        return;
      }

      $item = $this->getFieldsHelper()->createItem($index, $this->normalizeExternalReference($resultItem['external_reference']));
      $item->setScore($resultItem['relevance_score'] ?? 0);
      $results->addResultItem($item);
      $resultCount++;
    }

    $results->setResultCount($resultCount);
  }

  /**
   * {@inheritdoc}
   */
  public function isAvailable(): bool {
    $url = $this->configFactory->get('search_api_vragen_ai.settings')->get('vragen_ai_endpoint') ?? '';

    if (empty($url)) {
      $this->getLogger()->error("Vragen AI Endpoint is not set.");
      return FALSE;
    }

    if (filter_var($url, FILTER_VALIDATE_URL) === FALSE) {
      $this->getLogger()->error("Value set as Vragen AI Endpoint is invalid.");
      return FALSE;
    }

    return TRUE;
  }

  /**
   * Normalize external reference.
   *
   * @param string $externalReference
   *   The external reference to normalize.
   *
   * @return string
   *   The normalized external reference.
   */
  protected function normalizeExternalReference(string $externalReference): string {
    $siteIdentifier = mb_strtolower($this->configFactory->get('system.site')->get('name'));
    if (str_starts_with($externalReference, $siteIdentifier . ':')) {
      $externalReference = substr($externalReference, strlen($siteIdentifier) + 1);
    }

    return $externalReference;
  }

  /**
   * Get first document or create and return one.
   *
   * @param string $id
   *   The external reference ID.
   *
   * @return \Swis\JsonApi\Client\Interfaces\ItemInterface
   *   The document.
   */
  protected function firstOrCreateDocument(string $id): JsonApiItemInterface {
    try {
      $result = $this->documentRepository->all([
        'filter' => [
          'external_reference' => $id,
        ],
        'page' => ['size' => 1, 'number' => 1],
      ]);

      if ($result instanceof CollectionDocumentInterface && $item = $result->getData()->first()) {
        /** @var \Swis\JsonApi\Client\Interfaces\ItemInterface $item */
        return $item;
      }
    }
    catch (ValidationException $e) {
      // When response failed to validate, we can safely assume
      // that the document doesn't exist yet.
      $this->getLogger()->notice('Failed to receive existing document: ' . $e->getMessage());
    }

    $itemHydrator = new ItemHydrator(Client::getTypeMapper());
    return $itemHydrator->hydrate(
      Client::getTypeMapper()->getMapping('documents'),
      ['external_reference' => $id]
    );
  }

  /**
   * Get the index content for the item.
   *
   * @param \Drupal\search_api\Item\ItemInterface $item
   *   The item to get the content for.
   * @param array $metadata
   *   The metadata array for the item.
   *
   * @return string
   *   The index content.
   */
  protected function getIndexContentForItem(ItemInterface $item, array &$metadata): string {
    $contentFields = [];
    foreach ($item->getFields() as $field) {
      $fieldValues = match($field->getType()) {
        'vragen_ai_metadata', 'text', 'string', 'integer' => $field->getValues(),
        'boolean' => array_map(fn ($value) => sprintf('%s: %s', $field->getLabel(), $value ? 'true' : 'false'), $field->getValues()),
        default => NULL,
      };

      if (empty($fieldValues)) {
        continue;
      }

      if ($field->getType() === 'vragen_ai_metadata') {
        if (count($fieldValues) > 1) {
          $metadata[$field->getFieldIdentifier()] = $fieldValues;
        }
        else {
          $metadata[$field->getFieldIdentifier()] = $fieldValues[0];
        }
        continue;
      }

      $fieldValues = array_map(fn ($value) => $this->hasBlockLevelElements($value) ? $value : sprintf('<p>%s</p>', $value), $fieldValues);
      $contentFields[] = implode("\n", $fieldValues);
    }

    return sprintf('<!doctype html><html><body>%s</body></html>', implode("\n", $contentFields));
  }

  /**
   * Checks if the content has block level elements.
   *
   * @param string $content
   *   The content to check.
   *
   * @return bool
   *   TRUE if the content has block level elements, FALSE otherwise.
   */
  protected function hasBlockLevelElements(string $content): bool {
    return strip_tags($content) !== strip_tags($content, '<a><b><i><strong><em><span><button><br><code><abbr><sup><sub><mark><small><time>');
  }

  /**
   * Returns the search phrase from the keys.
   *
   * @param array $keys
   *   The search keys to build the search phrase from.
   *
   * @return string
   *   The search phrase.
   */
  protected function getSearchPhrase(array $keys): string {
    $words = [];

    foreach ($keys as $i => $key) {
      if (!Element::child($i)) {
        continue;
      }
      if (is_scalar($key)) {
        $words[] = $key;
      }
    }

    return implode(' ', $words);
  }

  /**
   * {@inheritdoc}
   */
  public function __sleep(): array {
    $properties = array_flip(parent::__sleep());
    unset($properties['documentRepository']);
    return array_keys($properties);
  }

  /**
   * {@inheritdoc}
   */
  public function __wakeup(): void {
    parent::__wakeup();

    $this->withDocumentRepository(self::buildDefaultDocumentRepository($this->configFactory));
  }

}
