<?php

namespace Drupal\search_api_vragen_ai\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\search_api_vragen_ai\Client;
use Drupal\search_api_vragen_ai\Client\SystemRepository;
use Http\Discovery\Psr18ClientDiscovery;
use Swis\JsonApi\Client\DocumentFactory;

/**
 * Configure settings for 'vragen.ai' integration.
 */
class VragenAiSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'search_api_vragen_ai_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['search_api_vragen_ai.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $vragen_ai_settings = $this->config('search_api_vragen_ai.settings');

    $form['endpoint'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Endpoint'),
      '#placeholder' => 'https://[account].vragen.ai/api/v1',
      '#description' => $this->t('Endpoint URL for Vragen.ai. More information on <a href="https://www.vragen.ai/docs">Vragen.ai</a>.'),
      '#default_value' => $vragen_ai_settings->get('vragen_ai_endpoint') ?? '',
      '#required' => TRUE,
      '#ajax' => [
        'callback' => '::updateSystemsDropdown',
        'wrapper' => 'vragen-ai-systems-wrapper',
        'event' => 'change',
        'progress' => ['type' => 'throbber'],
      ],
    ];

    $form['token'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Bearer token'),
      '#description' => $this->t('Enables access to the Vragen.ai API. More information on <a href="https://www.vragen.ai/docs">Vragen.ai</a>'),
      '#default_value' => $vragen_ai_settings->get('vragen_ai_token') ?? '',
      '#required' => TRUE,
      '#ajax' => [
        'callback' => '::updateSystemsDropdown',
        'wrapper' => 'vragen-ai-systems-wrapper',
        'event' => 'change',
        'progress' => ['type' => 'throbber'],
      ],
    ];

    // AJAX wrapper for systems dropdown.
    $form['systems_container'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'vragen-ai-systems-wrapper'],
    ];

    try {
      $endpoint = $form_state->getValue('endpoint') ?? $vragen_ai_settings->get('vragen_ai_endpoint');
      $token = $form_state->getValue('token') ?? $vragen_ai_settings->get('vragen_ai_token');

      if (!empty($endpoint) && !empty($token)) {
        $systems = $this->getSystemsRepository($endpoint, $token)->all([
          'filter' => [
            'system_type' => 'searchSystem',
          ],
        ]);
        if ($systems->getErrors()->isNotEmpty()) {
          throw new \Exception($systems->getErrors()->first()->getDetail());
        }

        $options = [];

        foreach ($systems->getData() as $system) {
          $options[$system->endpoint] = $system->tag ?? $system->name ?? $system->id;
        }

        $form['systems_container']['search_system'] = [
          '#type' => 'select',
          '#title' => $this->t('Search System'),
          '#description' => $this->t('Select the Vragen.ai search system to use when searching.'),
          '#options' => $options,
          '#default_value' => $vragen_ai_settings->get('vragen_ai_search_endpoint') ?? '',
          '#empty_option' => $this->t('- None -'),
        ];
      }
      else {
        $form['systems_container']['search_system'] = [
          '#markup' => '<div class="messages messages--warning">' . $this->t('Enter endpoint and token to load search systems.') . '</div>',
        ];
      }
    }
    catch (\Exception $e) {
      $form['systems_container']['search_system'] = [
        '#markup' => '<div class="messages messages--error">' . $this->t('Failed to load systems: @msg', ['@msg' => $e->getMessage()]) . '</div>',
      ];
    }

    $form['actions'] = [
      '#type' => 'actions',
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save'),
      '#button_type' => 'primary',
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $config = $this->config('search_api_vragen_ai.settings')
      ->set('vragen_ai_endpoint', $form_state->getValue('endpoint'))
      ->set('vragen_ai_token', $form_state->getValue('token'));

    if ($form_state->hasValue('search_system')) {
      $config->set('vragen_ai_search_endpoint', $form_state->getValue('search_system'));
    }

    $config->save();
    $this->messenger()->addMessage($this->t('The configuration options have been saved.'));
    parent::submitForm($form, $form_state);
  }

  /**
   * AJAX callback to rebuild the systems dropdown.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The updated systems container.
   */
  public function updateSystemsDropdown(array &$form, FormStateInterface $form_state): array {
    return $form['systems_container'];
  }

  /**
   * Get the systems repository.
   *
   * @param string|null $endpoint
   *   The API endpoint.
   * @param string|null $token
   *   The API token.
   *
   * @return \Drupal\search_api_vragen_ai\Client\SystemRepository
   *   The systems repository.
   */
  protected function getSystemsRepository(?string $endpoint = NULL, ?string $token = NULL): SystemRepository {
    $httpClient = Psr18ClientDiscovery::find();

    $apiEndpoint = $endpoint ?? $this->config('search_api_vragen_ai.settings')->get('vragen_ai_endpoint');
    $apiToken = $token ?? $this->config('search_api_vragen_ai.settings')->get('vragen_ai_token');

    $client = Client::create(Client::getTypeMapper(), $httpClient, $apiEndpoint, $apiToken);

    return new SystemRepository(
      $client,
      new DocumentFactory()
    );
  }

}
