<?php

namespace Drupal\search_api_vragen_ai\Client;

use Drupal\Core\Cache\Cache;
use Drupal\Core\Cache\CacheBackendInterface;
use Psr\Http\Client\ClientInterface as HttpClientInterface;
use Swis\JsonApi\Client\Client as JsonApiClient;
use Swis\JsonApi\Client\DocumentClient;
use Swis\JsonApi\Client\DocumentFactory;
use Swis\JsonApi\Client\Interfaces\TypeMapperInterface;
use Swis\JsonApi\Client\Parsers\ResponseParser;
use Swis\JsonApi\Client\TypeMapper;

/**
 * Client to interact with the Vragen.ai API.
 */
class Client {
  /**
   * The JSON API client.
   */
  protected JsonApiClient $jsonApiClient;

  /**
   * The JSON API document factory.
   */
  protected DocumentFactory $documentFactory;

  /**
   * The JSON API type mapper.
   */
  protected TypeMapperInterface $typeMapper;

  /**
   * The JSON API document client.
   */
  protected DocumentClient $documentClient;

  /**
   * The cache backend.
   *
   * This is used to cache system data to reduce API calls.
   */
  protected CacheBackendInterface $cacheBackend;

  /**
   * The cache key for this client.
   *
   * @var string
   */
  protected string $cacheKey;

  /**
   * Static cache of repositories.
   *
   * @var array<string, \Swis\JsonApi\Client\BaseRepository>
   */
  protected array $repositories = [];

  /**
   * Construct a Vragen.ai API client.
   *
   * It is highly recommended to use the ClientFactory to create instances of
   * this class.
   *
   * @param \Psr\Http\Client\ClientInterface $httpClient
   *   The HTTP client to use for API requests. We assume this client has a
   *   base URI set to the Vragen.ai API endpoint and includes the necessary
   *   authentication headers.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cacheBackend
   *   The cache backend to use for caching system data.
   * @param string $cacheKey
   *   The cache key to use for this client.
   */
  public function __construct(HttpClientInterface $httpClient, CacheBackendInterface $cacheBackend, string $cacheKey) {
    $this->jsonApiClient = new JsonApiClient($httpClient);
    $this->documentFactory = new DocumentFactory();
    $this->typeMapper = $this->buildTypeMapper();
    $this->documentClient = new DocumentClient($this->jsonApiClient, ResponseParser::create($this->typeMapper));

    $this->cacheBackend = $cacheBackend;
    $this->cacheKey = $cacheKey;
  }

  /**
   * Build the type mapper for the JSON API document client.
   *
   * @return \Swis\JsonApi\Client\Interfaces\TypeMapperInterface
   *   The type mapper instance.
   */
  protected function buildTypeMapper(): TypeMapperInterface {
    $typeMapper = new TypeMapper();

    $typeMapper->setMapping('documents', DocumentItem::class);
    $typeMapper->setMapping('systems', SystemItem::class);

    return $typeMapper;
  }

  /**
   * Get the document client.
   *
   * @return \Swis\JsonApi\Client\DocumentClient
   *   The document client instance.
   */
  public function getDocumentClient(): DocumentClient {
    return $this->documentClient;
  }

  /**
   * Get the document factory.
   *
   * @return \Swis\JsonApi\Client\DocumentFactory
   *   The document factory instance.
   */
  public function getDocumentFactory(): DocumentFactory {
    return $this->documentFactory;
  }

  /**
   * Get the cache backend.
   *
   * @return \Drupal\Core\Cache\CacheBackendInterface
   *   The cache backend instance.
   */
  public function getCacheBackend(): CacheBackendInterface {
    return $this->cacheBackend;
  }

  /**
   * Get the cache key for this client.
   *
   * @return string
   *   The cache key.
   */
  public function getCacheKey(): string {
    return $this->cacheKey;
  }

  /**
   * Get the document repository.
   *
   * @return \Drupal\search_api_vragen_ai\Client\DocumentRepository
   *   The document repository instance.
   */
  public function documents(): DocumentRepository {
    if (isset($this->repositories['documents']) && $this->repositories['documents'] instanceof DocumentRepository) {
      return $this->repositories['documents'];
    }

    $this->repositories['documents'] = new DocumentRepository($this);

    return $this->repositories['documents'];
  }

  /**
   * Get the system repository.
   *
   * @return \Drupal\search_api_vragen_ai\Client\SystemRepository
   *   The system repository instance.
   */
  public function systems(): SystemRepository {
    if (isset($this->repositories['systems']) && $this->repositories['systems'] instanceof SystemRepository) {
      return $this->repositories['systems'];
    }

    $this->repositories['systems'] = new SystemRepository($this);

    return $this->repositories['systems'];
  }

  /**
   * Search for documents in a specific system.
   *
   * @param \Drupal\search_api_vragen_ai\Client\SystemItem $system
   *   The system to search in.
   * @param string $query
   *   The search query to execute.
   *
   * @return \Drupal\search_api_vragen_ai\Client\SearchResultCollection
   *   A collection of search results.
   *
   * @throws \RuntimeException
   *   If the search fails or the response is invalid.
   * @throws \Psr\Http\Client\ClientExceptionInterface
   *   If there is an error with the HTTP client.
   */
  public function search(SystemItem $system, string $query): SearchResultCollection {
    $endpoint = $system->endpoint . (str_contains($system->endpoint, '?') ? '&' : '?') . http_build_query([
      'query' => $query,
    ]);

    $response = $this->jsonApiClient->get($endpoint, [
      'Accept' => 'application/json',
    ]);

    if ($response->getStatusCode() !== 200) {
      throw new \RuntimeException(sprintf('Failed to search in system %s: %s', $system->id, $response->getReasonPhrase()));
    }

    $data = json_decode($response->getBody()->getContents(), TRUE);
    if (!isset($data['results'])) {
      throw new \RuntimeException(sprintf('Invalid response from system %s: %s', $system->id, json_encode($data)));
    }

    $results = new SearchResultCollection($data['run_reference']);
    foreach ($data['results'] as $result) {
      $searchResult = new SearchResult($result);
      $results->add($searchResult);
    }

    return $results;
  }

  /**
   * Clear the cache for this client.
   */
  public function clearCache(): void {
    Cache::invalidateTags(['search_api_vragen_ai_client:' . $this->getCacheKey()]);
    $this->systems()->clearStaticCache();
  }

}
