<?php

namespace Drupal\search_api_vragen_ai\Client;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Http\ClientFactory as HttpClientFactory;
use Psr\Http\Client\ClientInterface as HttpClientInterface;

/**
 * Factory class to create instances of the Vragen.ai API client.
 */
class ClientFactory {
  /**
   * The HTTP client factory.
   */
  protected HttpClientFactory $httpClientFactory;

  /**
   * The cache backend.
   *
   * This is used to cache system data to reduce API calls.
   */
  protected CacheBackendInterface $cacheBackend;

  /**
   * Construct a new ClientFactory.
   *
   * @param \Drupal\Core\Http\ClientFactory $httpClientFactory
   *   The HTTP client factory.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cacheBackend
   *   The cache backend to use for caching API responses.
   */
  public function __construct(HttpClientFactory $httpClientFactory, CacheBackendInterface $cacheBackend) {
    $this->httpClientFactory = $httpClientFactory;
    $this->cacheBackend = $cacheBackend;
  }

  /**
   * Create a new instance of the Vragen.ai API client.
   *
   * @param string $endpoint
   *   The API endpoint URL.
   * @param string $token
   *   The API token for authentication.
   *
   * @return \Drupal\search_api_vragen_ai\Client\Client
   *   The configured Vragen.ai API client.
   */
  public function createClient(string $endpoint, string $token): Client {
    $endpoint = rtrim($endpoint, '/') . '/';

    $cacheKey = $this->getCacheKey($endpoint, $token);

    return new Client(
      $this->createHttpClient($endpoint, $token),
      $this->cacheBackend,
      $cacheKey
    );
  }

  /**
   * Generate a cache key based on the endpoint and token.
   *
   * @param string $endpoint
   *   The API endpoint URL.
   * @param string $token
   *   The API token for authentication.
   *
   * @return string
   *   The generated cache key.
   */
  protected function getCacheKey(string $endpoint, string $token): string {
    return sha1($endpoint . '|' . $token);
  }

  /**
   * Create a new HttpClient for the Vragen.ai API.
   *
   * @param string $endpoint
   *   The API endpoint URL.
   * @param string $token
   *   The API token for authentication.
   *
   * @return \Psr\Http\Client\ClientInterface
   *   The configured HTTP client.
   */
  public function createHttpClient(string $endpoint, string $token): HttpClientInterface {
    $endpoint = rtrim($endpoint, '/') . '/';

    return $this->httpClientFactory->fromOptions([
      'base_uri' => $endpoint,
      'headers' => [
        'Authorization' => 'Bearer ' . $token,
      ],
    ]);
  }

}
