<?php

namespace Drupal\search_api_vragen_ai\Client;

use Drupal\search_api\LoggerTrait;
use Swis\JsonApi\Client\Actions\Create;
use Swis\JsonApi\Client\Actions\Delete;
use Swis\JsonApi\Client\Actions\Update;
use Swis\JsonApi\Client\Exceptions\ValidationException;
use Swis\JsonApi\Client\Interfaces\CollectionDocumentInterface;

/**
 * Document repository containing methods to interact with the Vragen.ai API.
 *
 * @extends \Drupal\search_api_vragen_ai\Client\BaseRepository<\Drupal\search_api_vragen_ai\Client\DocumentItem>
 */
class DocumentRepository extends BaseRepository {
  use Delete;
  /* @use \Swis\JsonApi\Client\Actions\Update<\Drupal\search_api_vragen_ai\Client\DocumentItem> */
  use Update;
  /* @use \Swis\JsonApi\Client\Actions\Create<\Drupal\search_api_vragen_ai\Client\DocumentItem> */
  use Create;

  use LoggerTrait;

  /**
   * {@inheritdoc}
   */
  protected $endpoint = 'documents';

  /**
   * Find a document by its external reference.
   *
   * @param string $externalReference
   *   The external reference to search for.
   *
   * @return \Drupal\search_api_vragen_ai\Client\DocumentItem|null
   *   The DocumentItem if found, or NULL if not found.
   */
  public function findByExternalReference(string $externalReference): ?DocumentItem {
    try {
      $result = $this->all([
        'filter' => [
          'external_reference' => $externalReference,
        ],
        'page' => ['size' => 1, 'number' => 1],
      ]);

      if ($result instanceof CollectionDocumentInterface && $item = $result->getData()->first()) {
        return $item;
      }
    }
    catch (ValidationException $e) {
      // When response failed to validate, we can safely assume
      // that the document doesn't exist yet.
      $this->getLogger()->notice('Failed to receive existing document: ' . $e->getMessage());
    }

    return NULL;
  }

  /**
   * Create a new DocumentItem instance with the given external reference.
   *
   * @param string $externalReference
   *   The external reference to set for the new document.
   *
   * @return \Drupal\search_api_vragen_ai\Client\DocumentItem
   *   A new DocumentItem instance with the external reference set.
   */
  public function newByExternalReference(string $externalReference): DocumentItem {
    $document = new DocumentItem();
    $document->setAttribute('external_reference', $externalReference);
    return $document;
  }

}
