<?php

namespace Drupal\search_api_vragen_ai\Client;

/**
 * Class to represent a search result.
 */
class SearchResult {
  /**
   * The attributes of the search result.
   *
   * @var array<string, mixed>
   */
  protected array $attributes = [];

  /**
   * Constructor to initialize the search result with attributes.
   *
   * @param array<string, mixed> $attributes
   *   An associative array of attributes for the search result.
   */
  public function __construct(array $attributes = []) {
    $this->attributes = $attributes;
  }

  /**
   * Get the attributes of the search result.
   *
   * @return array<string, mixed>
   *   An associative array of attributes.
   */
  public function getAttributes(): array {
    return $this->attributes;
  }

  /**
   * Check if the search result has a specific attribute.
   *
   * @param string $name
   *   The name of the attribute to check.
   *
   * @return bool
   *   TRUE if the attribute exists, FALSE otherwise.
   */
  public function hasAttribute(string $name): bool {
    return array_key_exists($name, $this->attributes);
  }

  /**
   * Get a specific attribute from the search result.
   *
   * @param string $name
   *   The name of the attribute to retrieve.
   *
   * @return mixed
   *   The value of the attribute if it exists, NULL otherwise.
   */
  public function getAttribute(string $name): mixed {
    if ($this->hasAttribute($name)) {
      return $this->attributes[$name];
    }
    return NULL;
  }

  /**
   * Magic method to get an attribute as a property.
   *
   * @param string $name
   *   The name of the attribute to retrieve.
   *
   * @return mixed
   *   The value of the attribute if it exists, NULL otherwise.
   */
  public function __get(string $name): mixed {
    return $this->getAttribute($name);
  }

}
