<?php

namespace Drupal\search_api_vragen_ai\Client;

/**
 * Class to represent a collection of search results.
 */
class SearchResultCollection implements \IteratorAggregate {
  /**
   * The reference for the run that generated this collection.
   */
  protected string $runReference;

  /**
   * The search results in this collection.
   *
   * @var SearchResult[]
   */
  protected array $results = [];

  /**
   * Constructor for SearchResultCollection.
   *
   * @param string $runReference
   *   The reference for the run that generated this collection.
   * @param SearchResult[] $results
   *   An array of SearchResult objects.
   */
  public function __construct(string $runReference, array $results = []) {
    $this->runReference = $runReference;

    foreach ($results as $result) {
      $this->add($result);
    }
  }

  /**
   * Add a search result to the collection.
   *
   * @param \Drupal\search_api_vragen_ai\Client\SearchResult $result
   *   The search result to add.
   */
  public function add(SearchResult $result): void {
    $this->results[] = $result;
  }

  /**
   * Get all search results in the collection.
   *
   * @return SearchResult[]
   *   An array of SearchResult objects.
   */
  public function getResults(): array {
    return $this->results;
  }

  /**
   * Get the reference for the run that generated this collection.
   *
   * @return string
   *   The run reference.
   */
  public function getRunReference(): string {
    return $this->runReference;
  }

  /**
   * Get a slice of the search results based on offset and limit.
   *
   * @param int $offset
   *   The offset to start the slice from.
   * @param int|null $limit
   *   The maximum number of results to return. If NULL, returns all results.
   *
   * @return \Drupal\search_api_vragen_ai\Client\SearchResultCollection
   *   A new SearchResultCollection containing the sliced results.
   */
  public function getRange(int $offset, ?int $limit = NULL): SearchResultCollection {
    if (isset($limit)) {
      $slice = array_slice($this->results, $offset, $limit);
    }
    else {
      $slice = array_slice($this->results, $offset);
    }

    return new SearchResultCollection($this->runReference, $slice);
  }

  /**
   * {@inheritdoc}
   */
  public function getIterator(): \Traversable {
    return new \ArrayIterator($this->results);
  }

  /**
   * Count the number of search results in the collection.
   *
   * @return int
   *   The number of search results in the collection.
   */
  public function count(): int {
    return count($this->results);
  }

}
