<?php

namespace Drupal\search_api_vragen_ai\Client;

use Drupal\Core\Cache\CacheBackendInterface;

/**
 * System repository containing methods to interact with the Vragen.ai API.
 *
 * @extends \Drupal\search_api_vragen_ai\Client\BaseRepository<\Drupal\search_api_vragen_ai\Client\SystemItem>
 */
class SystemRepository extends BaseRepository {

  /**
   * {@inheritdoc}
   */
  protected $endpoint = 'systems';

  /**
   * The cache backend.
   *
   * This is used to cache system data to reduce API calls.
   */
  protected CacheBackendInterface $cacheBackend;

  /**
   * The cache key for the client.
   *
   * @var string
   */
  protected string $cacheKey;

  /**
   * Static cache of systems by type.
   *
   * @var array<string, array<string, \Drupal\search_api_vragen_ai\Client\SystemItem>>|null
   */
  protected ?array $systemsCache = NULL;

  public function __construct(Client $client) {
    parent::__construct($client);

    $this->cacheBackend = $client->getCacheBackend();
    $this->cacheKey = $client->getCacheKey();
  }

  /**
   * Get all systems of a specific type, using caching to reduce API calls.
   *
   * @param string $type
   *   The type of systems to retrieve.
   * @param bool $forceRefresh
   *   Whether to force a refresh of the cache.
   *
   * @return array<string, \Drupal\search_api_vragen_ai\Client\SystemItem>
   *   A collection of SystemItem objects of the specified type.
   */
  public function cachedAllByType(string $type, $forceRefresh = FALSE): array {
    if ($this->systemsCache === NULL || $forceRefresh) {
      $cid = 'search_api_vragen_ai:systems:' . $this->cacheKey;

      if (!$forceRefresh && ($cached = $this->cacheBackend->get($cid))) {
        $this->systemsCache = $cached->data;
      }
      else {
        $allSystems = $this->allWithoutPagination();

        $this->systemsCache = [];
        foreach ($allSystems as $system) {
          /** @var \Drupal\search_api_vragen_ai\Client\SystemItem $system */
          $this->systemsCache[$system->system_type][$system->uuid] = $system;
        }

        $this->cacheBackend->set(
          $cid,
          $this->systemsCache,
          time() + 3600,
          [
            'search_api_vragen_ai_systems',
            'search_api_vragen_ai_client:' . $this->cacheKey,
          ]
        );
      }
    }

    return $this->systemsCache[$type] ?? [];
  }

  /**
   * Clear the static cache of systems.
   */
  public function clearStaticCache(): void {
    $this->systemsCache = NULL;
  }

}
