<?php

namespace Drupal\search_api_vragen_ai\Event;

use Drupal\Component\EventDispatcher\Event;
use Drupal\search_api\IndexInterface;
use Drupal\search_api\Item\ItemInterface;
use Drupal\search_api_vragen_ai\Client\DocumentItem;

/**
 * Event to be fired after a document has been created for indexing.
 *
 * This event allows you to modify the document before it is indexed.
 */
class PostCreateIndexDocumentEvent extends Event {
  /**
   * The Search API item to be indexed.
   */
  protected ItemInterface $item;

  /**
   * The document item to be indexed.
   */
  protected DocumentItem $document;

  /**
   * The Search API index.
   */
  protected IndexInterface $index;

  /**
   * Whether the item should be indexed.
   */
  protected bool $shouldIndex;

  /**
   * Constructs a new event instance.
   *
   * @param \Drupal\search_api\Item\ItemInterface $item
   *   The Search API item to be indexed.
   * @param \Drupal\search_api_vragen_ai\Client\DocumentItem $document
   *   The prepared Document item from the Search API item.
   * @param \Drupal\search_api\IndexInterface $index
   *   The Search API index.
   * @param bool $shouldIndex
   *   Whether the item should be indexed.
   */
  public function __construct(ItemInterface $item, DocumentItem $document, IndexInterface $index, bool $shouldIndex = TRUE) {
    $this->item = $item;
    $this->document = $document;
    $this->index = $index;
    $this->shouldIndex = $shouldIndex;
  }

  /**
   * Retrieves the Search API item to be indexed.
   *
   * @return \Drupal\search_api\Item\ItemInterface
   *   The Search API item to be indexed.
   */
  public function getItem(): ItemInterface {
    return $this->item;
  }

  /**
   * Retrieves the document item.
   *
   * @return \Drupal\search_api_vragen_ai\Client\DocumentItem
   *   The prepared Document item.
   */
  public function getDocument(): DocumentItem {
    return $this->document;
  }

  /**
   * Retrieves the index.
   *
   * @return \Drupal\search_api\IndexInterface
   *   The Search API index.
   */
  public function getIndex(): IndexInterface {
    return $this->index;
  }

  /**
   * Checks if the item should be indexed.
   *
   * @param bool|null $value
   *   Optional parameter to set the indexing status. If provided, it will
   *   update the internal state of whether the item should be indexed. If NULL,
   *   it will return the current state.
   *
   * @return bool
   *   TRUE if the item should be indexed, FALSE otherwise.
   */
  public function shouldIndex(?bool $value = NULL): bool {
    if ($value !== NULL) {
      $this->shouldIndex = (bool) $value;
    }

    return $this->shouldIndex;
  }

}
