<?php

namespace Drupal\search_api_vragen_ai\Client;

use Drupal\Core\Cache\Cache;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\search_api\Query\ConditionGroupInterface;
use Drupal\search_api_vragen_ai\FilterService;
use Psr\Http\Client\ClientInterface as HttpClientInterface;
use Swis\JsonApi\Client\Client as JsonApiClient;
use Swis\JsonApi\Client\Collection;
use Swis\JsonApi\Client\DocumentClient;
use Swis\JsonApi\Client\DocumentFactory;
use Swis\JsonApi\Client\Interfaces\TypeMapperInterface;
use Swis\JsonApi\Client\Item;
use Swis\JsonApi\Client\Parsers\ResponseParser;
use Swis\JsonApi\Client\TypeMapper;

/**
 * Client to interact with the Vragen.ai API.
 */
class Client {
  /**
   * The JSON API client.
   */
  protected JsonApiClient $jsonApiClient;

  /**
   * The JSON API document factory.
   */
  protected DocumentFactory $documentFactory;

  /**
   * The JSON API type mapper.
   */
  protected TypeMapperInterface $typeMapper;

  /**
   * The JSON API document client.
   */
  protected DocumentClient $documentClient;

  /**
   * The cache backend.
   *
   * This is used to cache system data to reduce API calls.
   */
  protected CacheBackendInterface $cacheBackend;

  /**
   * The cache key for this client.
   *
   * @var string
   */
  protected string $cacheKey;

  /**
   * Static cache of repositories.
   *
   * @var array<string, \Swis\JsonApi\Client\BaseRepository>
   */
  protected array $repositories = [];

  /**
   * Construct a Vragen.ai API client.
   *
   * It is highly recommended to use the ClientFactory to create instances of
   * this class.
   *
   * @param \Psr\Http\Client\ClientInterface $httpClient
   *   The HTTP client to use for API requests. We assume this client has a
   *   base URI set to the Vragen.ai API endpoint and includes the necessary
   *   authentication headers.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cacheBackend
   *   The cache backend to use for caching system data.
   * @param string $cacheKey
   *   The cache key to use for this client.
   */
  public function __construct(HttpClientInterface $httpClient, CacheBackendInterface $cacheBackend, string $cacheKey) {
    $this->jsonApiClient = new JsonApiClient($httpClient);
    $this->documentFactory = new DocumentFactory();
    $this->typeMapper = $this->buildTypeMapper();
    $this->documentClient = new DocumentClient($this->jsonApiClient, ResponseParser::create($this->typeMapper));

    $this->cacheBackend = $cacheBackend;
    $this->cacheKey = $cacheKey;
  }

  /**
   * Build the type mapper for the JSON API document client.
   *
   * @return \Swis\JsonApi\Client\Interfaces\TypeMapperInterface
   *   The type mapper instance.
   */
  protected function buildTypeMapper(): TypeMapperInterface {
    $typeMapper = new TypeMapper();

    $typeMapper->setMapping('documents', DocumentItem::class);

    return $typeMapper;
  }

  /**
   * Get the document client.
   *
   * @return \Swis\JsonApi\Client\DocumentClient
   *   The document client instance.
   */
  public function getDocumentClient(): DocumentClient {
    return $this->documentClient;
  }

  /**
   * Get the document factory.
   *
   * @return \Swis\JsonApi\Client\DocumentFactory
   *   The document factory instance.
   */
  public function getDocumentFactory(): DocumentFactory {
    return $this->documentFactory;
  }

  /**
   * Get the cache backend.
   *
   * @return \Drupal\Core\Cache\CacheBackendInterface
   *   The cache backend instance.
   */
  public function getCacheBackend(): CacheBackendInterface {
    return $this->cacheBackend;
  }

  /**
   * Get the cache key for this client.
   *
   * @return string
   *   The cache key.
   */
  public function getCacheKey(): string {
    return $this->cacheKey;
  }

  /**
   * Get the document repository.
   *
   * @return \Drupal\search_api_vragen_ai\Client\DocumentRepository
   *   The document repository instance.
   */
  public function documents(): DocumentRepository {
    if (isset($this->repositories['documents']) && $this->repositories['documents'] instanceof DocumentRepository) {
      return $this->repositories['documents'];
    }

    $this->repositories['documents'] = new DocumentRepository($this);

    return $this->repositories['documents'];
  }

  /**
   * Get the system repository.
   *
   * @return \Drupal\search_api_vragen_ai\Client\SystemRepository
   *   The system repository instance.
   */
  public function systems(): SystemRepository {
    if (isset($this->repositories['systems']) && $this->repositories['systems'] instanceof SystemRepository) {
      return $this->repositories['systems'];
    }

    $this->repositories['systems'] = new SystemRepository($this);

    return $this->repositories['systems'];
  }

  /**
   * Search for documents in the Vragen.ai knowledge base.
   *
   * @param string $query
   *   The search query to execute.
   * @param \Drupal\search_api\Query\ConditionGroupInterface $conditions
   *   The SAPI condition group of the query,
   *   passing an empty condition group is allowed.
   * @param int $offset
   *   The number of documents to start pagination at.
   * @param int $limit
   *   The number of documents to return starting from the offset.
   * @param string|null $system_id
   *   An optional system ID if the search should be performed using a system.
   *
   * @return \Drupal\search_api_vragen_ai\Client\SearchResultCollection
   *   A collection of search results.
   *
   * @throws \RuntimeException
   *   If the search fails or the response is invalid.
   * @throws \Psr\Http\Client\ClientExceptionInterface
   *   If there is an error with the HTTP client.
   */
  public function search(
    string $query,
    ConditionGroupInterface $conditions,
    int $offset = 0,
    int $limit = 10,
    ?string $system_id = NULL,
  ): SearchResultCollection {
    $filter = FilterService::toQueryParams($conditions);
    $is_system_search = !empty($system_id);

    if ($is_system_search) {
      $response = $this->documents()->systemSearch($query, $system_id, $offset, $limit);
    }
    else {
      $response = $this->documents()->search($query, $offset, $limit, $filter);
    }

    $results = new SearchResultCollection($response->getMeta()?->toArray() ?? []);
    $data = $response->getData();
    if (!$data instanceof Collection) {
      return $results;
    }
    foreach ($data->all() as $document) {
      assert($document instanceof Item);
      $searchResult = new SearchResult($document->attributesToArray());
      $results->add($searchResult);
    }

    if ($is_system_search) {
      return $results->getRange($offset, $limit);
    }

    return $results;
  }

  /**
   * Search for similar documents in the Vragen.ai knowledge base.
   *
   * @param string $external_id
   *   The SAPI external_id of the document to find similar documents for.
   * @param \Drupal\search_api\Query\ConditionGroupInterface $conditions
   *   The SAPI condition group of the query,
   *   passing an empty condition group is allowed.
   * @param int $offset
   *   The number of documents to start pagination at.
   * @param int $limit
   *   The number of documents to return starting from the offset.
   *
   * @return SearchResultCollection
   *   A collection of search results.
   */
  public function similar(
    string $external_id,
    ConditionGroupInterface $conditions,
    int $offset = 0,
    int $limit = 10,
  ): SearchResultCollection {
    $filter = FilterService::toQueryParams($conditions);
    $response = $this->documents()->similar($external_id, $offset, $limit, $filter);

    $results = new SearchResultCollection($response->getMeta()?->toArray() ?? []);
    $data = $response->getData();
    if (!$data instanceof Collection) {
      return $results;
    }
    foreach ($data->all() as $document) {
      assert($document instanceof Item);
      $searchResult = new SearchResult($document->attributesToArray());
      $results->add($searchResult);
    }

    return $results;
  }

  /**
   * Clear the cache for this client.
   */
  public function clearCache(): void {
    Cache::invalidateTags(['search_api_vragen_ai_client:' . $this->getCacheKey()]);
  }

}
