<?php

namespace Drupal\search_api_vragen_ai\Client;

use Drupal\search_api\LoggerTrait;
use Swis\JsonApi\Client\Actions\Create;
use Swis\JsonApi\Client\Actions\Delete;
use Swis\JsonApi\Client\Actions\Update;
use Swis\JsonApi\Client\Exceptions\ValidationException;
use Swis\JsonApi\Client\Interfaces\CollectionDocumentInterface;
use Swis\JsonApi\Client\Interfaces\DocumentInterface;

/**
 * Document repository containing methods to interact with the Vragen.ai API.
 *
 * @extends \Drupal\search_api_vragen_ai\Client\BaseRepository<\Drupal\search_api_vragen_ai\Client\DocumentItem>
 */
class DocumentRepository extends BaseRepository {
  use Delete;
  /* @use \Swis\JsonApi\Client\Actions\Update<\Drupal\search_api_vragen_ai\Client\DocumentItem> */
  use Update;
  /* @use \Swis\JsonApi\Client\Actions\Create<\Drupal\search_api_vragen_ai\Client\DocumentItem> */
  use Create;

  use LoggerTrait;

  /**
   * {@inheritdoc}
   */
  protected $endpoint = 'documents';

  /**
   * Find a document by its external reference.
   *
   * @param string $externalReference
   *   The external reference to search for.
   *
   * @return \Drupal\search_api_vragen_ai\Client\DocumentItem|null
   *   The DocumentItem if found, or NULL if not found.
   */
  public function findByExternalReference(string $externalReference): ?DocumentItem {
    try {
      $result = $this->all([
        'filter' => [
          'external_reference' => $externalReference,
        ],
        'page' => ['size' => 1, 'number' => 1],
      ]);

      if ($result instanceof CollectionDocumentInterface && $item = $result->getData()->first()) {
        return $item;
      }
    }
    catch (ValidationException $e) {
      // When response failed to validate, we can safely assume
      // that the document doesn't exist yet.
      $this->getLogger()->notice('Failed to receive existing document: ' . $e->getMessage());
    }

    return NULL;
  }

  /**
   * Create a new DocumentItem instance with the given external reference.
   *
   * @param string $externalReference
   *   The external reference to set for the new document.
   *
   * @return \Drupal\search_api_vragen_ai\Client\DocumentItem
   *   A new DocumentItem instance with the external reference set.
   */
  public function newByExternalReference(string $externalReference): DocumentItem {
    $document = new DocumentItem();
    $document->setAttribute('external_reference', $externalReference);
    return $document;
  }

  /**
   * Sends a search request to the Vragen.ai API.
   *
   * @param string $query
   *   The keyword to search on.
   * @param int $offset
   *   The offset to start the pagination at.
   * @param int $limit
   *   The amount of documents to retrieve starting from the offset.
   * @param array $filter
   *   Filter query parameters for filtering metadata or knowledge scopes.
   *
   * @return \Swis\JsonApi\Client\Interfaces\DocumentInterface
   *   A document interface that is likely to be a collection.
   */
  public function search(
    string $query,
    int $offset,
    int $limit,
    array $filter,
  ): DocumentInterface {
    $endpoint = "documents/search" . '?' . http_build_query([
      'query' => $query,
      'page[offset]' => $offset,
      'page[limit]' => $limit,
      'filter' => $filter,
    ]);
    $response = $this->getClient()->get($endpoint);

    if ($response->hasErrors()) {
      throw new \RuntimeException(sprintf('Failed to find search documents: %s', $response->getErrors()->first()->detail));
    }

    return $response;
  }

  /**
   * Sends a system search request to the Vragen.ai API.
   *
   * @param string $query
   *   The keyword to search on.
   * @param string $system_id
   *   The ID of the system to use for the search.
   * @param int $offset
   *   The offset to start the pagination at.
   * @param int $limit
   *   The amount of documents to retrieve starting from the offset.
   *
   * @return \Swis\JsonApi\Client\Interfaces\DocumentInterface
   *   A document interface that is likely to be a collection.
   */
  public function systemSearch(
    string $query,
    string $system_id,
    int $offset,
    int $limit,
  ): DocumentInterface {
    $endpoint = "systems/" . $system_id . "/search" . '?' . http_build_query([
      'query' => $query,
      'page[offset]' => $offset,
      'page[limit]' => $limit,
    ]);
    $response = $this->getClient()->get($endpoint);

    if ($response->hasErrors()) {
      throw new \RuntimeException(sprintf('Failed to find search documents: %s', $response->getErrors()->first()->detail));
    }

    return $response;
  }

  /**
   * Sends a similar request to the Vragen.ai API.
   *
   * @param string $external_id
   *   The external SAPI id of the document to find similar documents for.
   * @param int $offset
   *   The offset to start the pagination at.
   * @param int $limit
   *   The amount of documents to retrieve starting from the offset.
   * @param array $filter
   *   Filter query parameters for filtering metadata or knowledge scopes.
   *
   * @return \Swis\JsonApi\Client\Interfaces\DocumentInterface
   *   A document interface that is likely to be a collection.
   */
  public function similar(
    string $external_id,
    int $offset,
    int $limit,
    array $filter,
  ): DocumentInterface {
    $endpoint = "documents/similar" . '?' . http_build_query([
      'query' => $external_id,
      'page[offset]' => $offset,
      'page[limit]' => $limit,
      'filter' => $filter,
    ]);
    $response = $this->getClient()->get($endpoint);

    if ($response->hasErrors()) {
      throw new \RuntimeException(sprintf('Failed to find similar documents: %s', $response->getErrors()->first()->detail));
    }

    return $response;
  }

}
