<?php

namespace Drupal\search_api_vragen_ai\Client;

/**
 * Class to represent a collection of search results.
 */
class SearchResultCollection implements \IteratorAggregate {

  /**
   * The search results in this collection.
   *
   * @var SearchResult[]
   */
  protected array $results = [];

  /**
   * The total search results available.
   *
   * @var int
   */
  protected int $total;

  /**
   * The offset at which this page starts.
   *
   * @var int
   */
  protected int $offset;

  /**
   * The limit of returned search results.
   *
   * @var int
   */
  protected int $limit;

  /**
   * Constructor for SearchResultCollection.
   *
   * @param array[] $meta
   *   An array of JSON:API metadata.
   * @param SearchResult[] $results
   *   An array of SearchResult objects.
   */
  public function __construct(array $meta = [], array $results = []) {
    $this->total = $meta['total'] ?? count($results);
    $this->offset = $meta['offset'] ?? 0;
    $this->limit = $meta['limit'] ?? 0;

    foreach ($results as $result) {
      $this->add($result);
    }
  }

  /**
   * Add a search result to the collection.
   *
   * @param \Drupal\search_api_vragen_ai\Client\SearchResult $result
   *   The search result to add.
   */
  public function add(SearchResult $result): void {
    $this->results[] = $result;
  }

  /**
   * Get all search results in the collection.
   *
   * @return SearchResult[]
   *   An array of SearchResult objects.
   */
  public function getResults(): array {
    return $this->results;
  }

  /**
   * {@inheritdoc}
   */
  public function getIterator(): \Traversable {
    return new \ArrayIterator($this->results);
  }

  /**
   * Get a slice of the search results based on offset and limit.
   *
   * @param int $offset
   *   The offset to start the slice from.
   * @param int|null $limit
   *   The maximum number of results to return. If NULL, returns all results.
   *
   * @return \Drupal\search_api_vragen_ai\Client\SearchResultCollection
   *   A new SearchResultCollection containing the sliced results.
   */
  public function getRange(int $offset, ?int $limit = NULL): SearchResultCollection {
    if (isset($limit)) {
      $slice = array_slice($this->results, $offset, $limit);
    }
    else {
      $slice = array_slice($this->results, $offset);
    }

    return new SearchResultCollection(
      [
        'total' => $this->total,
        'offset' => $offset,
        'limit' => $limit ?? $this->limit,
      ],
      $slice
    );
  }

  /**
   * Count the number of search results in the collection.
   *
   * @return int
   *   The number of search results in the collection.
   */
  public function count(): int {
    return count($this->results);
  }

  /**
   * Get the total amount of search results available on Vragen.ai.
   */
  public function total(): int {
    return $this->total;
  }

  /**
   * Get the used offset for this search collection.
   */
  public function offset(): int {
    return $this->offset;
  }

  /**
   * Get the used limit for this search collection.
   */
  public function limit(): int {
    return $this->limit;
  }

}
