<?php

namespace Drupal\search_api_vragen_ai\EventSubscriber;

use Drupal\Core\Entity\Plugin\DataType\EntityAdapter;
use Drupal\file\Entity\File;
use Drupal\media\MediaInterface;
use Drupal\search_api_vragen_ai\Event\PostCreateIndexDocumentEvent;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Event subscriber for indexing media files.
 */
class IndexEventsSubscriber implements EventSubscriberInterface {

  /**
   * {@inheritDoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      PostCreateIndexDocumentEvent::class => 'onPostCreateIndexDocument',
    ];
  }

  /**
   * Handles the PostCreateIndexDocumentEvent.
   *
   * @param \Drupal\search_api_vragen_ai\Event\PostCreateIndexDocumentEvent $event
   *   The event containing the item to be indexed.
   */
  public function onPostCreateIndexDocument(PostCreateIndexDocumentEvent $event): void {
    $item = $event->getItem();

    /** @var \Drupal\Core\Entity\Plugin\DataType\EntityAdapter|\Drupal\Core\Entity\EntityInterface|null $entity */
    $entity = $item->getOriginalObject();
    if ($entity instanceof EntityAdapter) {
      $entity = $entity->getEntity();
    }

    if (!$entity instanceof MediaInterface) {
      return;
    }

    $mediaType = $entity->bundle->entity;

    $fieldDefinition = $entity->getSource()->getSourceFieldDefinition($mediaType);
    if ($fieldDefinition->getType() !== 'file') {
      $event->shouldIndex(FALSE);
      return;
    }

    $fileId = $entity->getSource()->getSourceFieldValue($entity);
    $file = File::load($fileId);

    if (!$file || $file->getMimeType() !== 'application/pdf') {
      $event->shouldIndex(FALSE);
      return;
    }

    $document = $event->getDocument();
    $document->fill([
      'content' => NULL,
      'url' => $file->createFileUrl(FALSE),
      'mime_type' => $file->getMimeType(),
    ]);
  }

}
