<?php

namespace Drupal\search_api_vragen_ai;

use Drupal\search_api\Query\ConditionGroupInterface;
use Drupal\search_api\Query\ConditionInterface;

/**
 * Converts Drupal Search API ConditionGroups to Vragen.ai query strings.
 *
 * Supports nested ConditionGroups and most operators.
 */
final class FilterService {

  /**
   * Convert a Drupal Search API condition tree into Vragen.ai query params.
   *
   *  Filter[0][id]
   *  filter[0][parent]
   *  filter[0][path]
   *  filter[0][value]
   *  filter[0][operator]
   *
   * @param \Drupal\search_api\Query\ConditionGroupInterface $root
   *   The ConditionGroup to convert.
   *
   * @return array
   *   The query parameters from the ConditionGroup.
   */
  public static function toQueryParams(ConditionGroupInterface $root): array {
    $state = [
      'groupCounter' => 0,
      'rows' => [],
    ];

    $rootGroupId = self::newGroup($state, strtoupper($root->getConjunction()), 'root');
    self::walkGroup($state, $root, $rootGroupId);

    $out = [];

    foreach ($state['rows'] as $index => $row) {
      if (array_key_exists('id', $row)) {
        $out["filter[$index][id]"] = $row['id'];
      }
      if (array_key_exists('parent', $row)) {
        $out["filter[$index][parent]"] = $row['parent'];
      }
      if (array_key_exists('path', $row)) {
        $out["filter[$index][path]"] = $row['path'];
      }
      if (array_key_exists('operator', $row)) {
        $out["filter[$index][operator]"] = $row['operator'];
      }
      if (array_key_exists('value', $row)) {
        $out["filter[$index][value]"] = $row['value'];
      }
    }

    return $out;
  }

  /**
   * Walks through a ConditionGroup converting all conditions.
   *
   * @param array $state
   *   Internal state used while building the query params.
   * @param \Drupal\search_api\Query\ConditionGroupInterface $group
   *   The ConditionGroup to transform.
   * @param string $parentGroupId
   *   The parent ID of this ConditionGroup.
   *
   * @return void
   *   Nothing.
   */
  private static function walkGroup(
    array &$state,
    ConditionGroupInterface $group,
    string $parentGroupId,
  ): void {
    foreach ($group->getConditions() as $child) {
      if ($child instanceof ConditionGroupInterface) {
        $childGroupId = self::newGroup(
          $state,
          strtoupper($child->getConjunction()),
          $parentGroupId
        );
        self::walkGroup($state, $child, $childGroupId);
        continue;
      }

      if ($child instanceof ConditionInterface) {
        self::emitCondition($state, $child, $parentGroupId);
      }
    }
  }

  /**
   * Emits a query param condition in exchange for a ConditionInterface.
   *
   * @param array $state
   *   Internal state used while building the query params.
   * @param \Drupal\search_api\Query\ConditionInterface $condition
   *   The ConditionInterface to transform.
   * @param string $currentGroupId
   *   The ConditionGroup to emit for.
   *
   * @return void
   *   Nothing.
   */
  private static function emitCondition(
    array &$state,
    ConditionInterface $condition,
    string $currentGroupId,
  ): void {
    $field = $condition->getField();
    $rawValue = $condition->getValue();
    $rawOperator = strtoupper($condition->getOperator());
    $operator = self::mapOperator($rawOperator);

    if (is_array($rawValue) && in_array($operator, ['eq', 'like'], TRUE)) {
      $orGroupId = self::newGroup($state, 'OR', $currentGroupId);
      foreach ($rawValue as $v) {
        self::addFilterRow($state, $orGroupId, $field, $operator, self::toScalar($v));
      }
      return;
    }

    $value = $operator === 'IS NULL'
      ? self::normalizeIsNullValue($rawOperator, $rawValue)
      : self::normalizeValue($operator, $rawValue);

    self::addFilterRow($state, $currentGroupId, $field, $operator, $value);
  }

  /**
   * Add a leaf row.
   *
   *  This becomes a row like:
   *    [
   *      'parent'   => '1',
   *      'path'     => 'foo',
   *      'operator' => 'eq',
   *      'value'    => 'bar',
   *    ]
   *
   * @param array $state
   *   Internal state used while building the query params.
   * @param string $groupId
   *   The ID of the group this leaf belongs to.
   * @param string $path
   *   The path of this leaf.
   * @param string $operator
   *   The operator of this leaf.
   * @param mixed $value
   *   The value this leaf will filter on.
   *
   * @return void
   *   Nothing.
   */
  private static function addFilterRow(
    array &$state,
    string $groupId,
    string $path,
    string $operator,
    mixed $value,
  ): void {
    $state['rows'][] = [
      'parent' => $groupId,
      'path' => $path,
      'operator' => $operator,
      'value' => self::toScalar($value),
    ];
  }

  /**
   * Add a group row.
   *
   *  This becomes a row like:
   *    [
   *      'id'       => '1',
   *      'parent'   => 'root' | '<parentGroupId>',
   *      'operator' => 'AND' | 'OR',
   *    ]
   *
   * @param array $state
   *   Internal state used while building the query params.
   * @param string $operator
   *   The operator for this group.
   * @param string $parent
   *   The parent of this group.
   *
   * @return string
   *   The ID of the newly generated group.
   */
  private static function newGroup(array &$state, string $operator, string $parent): string {
    $id = (string) ++$state['groupCounter'];

    $state['rows'][] = [
      'id' => $id,
      'parent' => $parent,
      'operator' => ($operator === 'OR' ? 'OR' : 'AND'),
    ];

    return $id;
  }

  /**
   * Maps a SAPI operator to one compatible with Vragen.ai.
   *
   * @param string $operator
   *   The SAPI operator.
   *
   * @return string
   *   Vragen.ai compatible operator.
   */
  private static function mapOperator(string $operator): string {
    return match ($operator) {
      '!=', '<>'  => '!=',
      '>', 'GT'  => '>',
      '>=', 'GTE' => '>=',
      '<', 'LT' => '<',
      '<=', 'LTE' => '<=',
      'LIKE', 'CONTAINS'  => 'LIKE',
      'IS NULL', 'IS NOT NULL' => 'IS NULL',
      'CONTAINS_ANY' => 'IN',
      'CONTAINS_NONE' => 'NOT IN',
      'CONTAINS_ALL' => 'ALL',
      default => '=',
    };
  }

  /**
   * Normalizes the IS NOT NULL operator to an IS NULL with false.
   *
   * @param string $rawOperator
   *   The raw operator for the filter.
   * @param mixed $rawValue
   *   The raw value to filter on.
   *
   * @return bool
   *   The value of the IsNull filter.
   */
  private static function normalizeIsNullValue(string $rawOperator, mixed $rawValue): bool {
    if ($rawOperator === 'IS NOT NULL') {
      return FALSE;
    }
    elseif (is_bool($rawValue)) {
      return $rawValue;
    }
    if (is_string($rawValue)) {
      return strtolower($rawValue) !== 'false';
    }
    return $rawValue === NULL || $rawValue;
  }

  /**
   * Normalizes array values to proper query parameters.
   *
   * @param string $operator
   *   The operator of the filter.
   * @param mixed $value
   *   The value to filter on.
   *
   * @return mixed
   *   The normalized value.
   */
  private static function normalizeValue(string $operator, mixed $value): mixed {
    if (in_array($operator, ['IN', 'NOT IN', 'ALL']) && is_array($value)) {
      return implode(',', array_map('strval', $value));
    }

    return $value;
  }

  /**
   * Transforms any value into a scalar for use with filtering.
   *
   * @param mixed $value
   *   The value to transform.
   *
   * @return mixed
   *   The transformed scalar.
   */
  private static function toScalar(mixed $value): mixed {
    return match (TRUE) {
      is_scalar($value), $value === NULL => $value,
      default => (string) json_encode(
        $value,
        JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES
      ),
    };
  }

}
