<?php

namespace Drupal\search_api_vragen_ai\Plugin\search_api\backend;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\Plugin\DataType\EntityAdapter;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\PluginFormInterface;
use Drupal\Core\Render\Element;
use Drupal\search_api\Backend\BackendPluginBase;
use Drupal\search_api\IndexInterface;
use Drupal\search_api\Item\ItemInterface;
use Drupal\search_api\Plugin\PluginFormTrait;
use Drupal\search_api\Query\QueryInterface;
use Drupal\search_api\Utility\Utility as SearchApiUtility;
use Drupal\search_api_vragen_ai\Client\Client;
use Drupal\search_api_vragen_ai\Client\ClientFactory;
use Drupal\search_api_vragen_ai\Client\DocumentItem;
use Drupal\search_api_vragen_ai\Event\PostCreateIndexDocumentEvent;
use Drupal\user\EntityOwnerInterface;
use Swis\JsonApi\Client\Interfaces\CollectionDocumentInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Contracts\EventDispatcher\EventDispatcherInterface;

/**
 * The Vragen.ai search backend.
 *
 * @SearchApiBackend(
 *   id = "vragen_ai",
 *   label = @Translation("Vragen.ai"),
 *   description = @Translation("Vragen.ai semantic search.")
 * )
 */
class VragenAiBackend extends BackendPluginBase implements PluginFormInterface {

  use PluginFormTrait;

  /**
   * The Vragen.ai client factory.
   */
  protected ClientFactory $clientFactory;

  /**
   * The Vragen.ai client.
   */
  protected ?Client $client = NULL;

  /**
   * The event dispatcher.
   */
  protected EventDispatcherInterface $eventDispatcher;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);

    $instance->setEntityTypeManager($container->get('entity_type.manager'));
    $instance->setClientFactory($container->get('search_api_vragen_ai.client_factory'));
    $instance->setEventDispatcher($container->get('event_dispatcher'));

    return $instance;
  }

  /**
   * Set the client factory.
   *
   * @param \Drupal\search_api_vragen_ai\Client\ClientFactory $clientFactory
   *   The Vragen.ai client factory.
   */
  public function setClientFactory(ClientFactory $clientFactory): void {
    $this->clientFactory = $clientFactory;
  }

  /**
   * Set the event dispatcher.
   *
   * @param \Symfony\Contracts\EventDispatcher\EventDispatcherInterface $eventDispatcher
   *   The event dispatcher.
   */
  public function setEventDispatcher(EventDispatcherInterface $eventDispatcher): void {
    $this->eventDispatcher = $eventDispatcher;
  }

  /**
   * Set the entity type manager.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   */
  public function setEntityTypeManager(EntityTypeManagerInterface $entityTypeManager): void {
    $this->entityTypeManager = $entityTypeManager;
  }

  /**
   * Get the Vragen.ai client.
   *
   * @return \Drupal\search_api_vragen_ai\Client\Client
   *   The Vragen.ai client.
   */
  protected function getClient(): Client {
    if (isset($this->client)) {
      return $this->client;
    }

    if (empty($this->configuration['api_endpoint']) || empty($this->configuration['api_token'])) {
      throw new \Exception('Vragen.ai client is not configured.');
    }

    $this->client = $this->clientFactory->createClient($this->configuration['api_endpoint'], $this->configuration['api_token']);

    return $this->client;
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form['api_endpoint'] = [
      '#type' => 'textfield',
      '#title' => $this->t('API Endpoint'),
      '#default_value' => $this->configuration['api_endpoint'] ?? '',
      '#required' => TRUE,
      '#placeholder' => 'https://[account].vragen.ai/api/v1',
      '#description' => $this->t('Endpoint URL for Vragen.ai. More information on <a href="https://www.vragen.ai/docs">Vragen.ai</a>.'),
      '#ajax' => [
        'callback' => [static::class, 'updateSystemsDropdown'],
        'wrapper' => 'vragen-ai-systems-wrapper',
        'event' => 'change',
        'progress' => ['type' => 'throbber'],
      ],
    ];

    $form['api_token'] = [
      '#type' => 'textfield',
      '#title' => $this->t('API Bearer Token'),
      '#default_value' => $this->configuration['api_token'] ?? '',
      '#required' => TRUE,
      '#description' => $this->t('Enables access to the Vragen.ai API. More information on <a href="https://www.vragen.ai/docs">Vragen.ai</a>'),
      '#ajax' => [
        'callback' => [static::class, 'updateSystemsDropdown'],
        'wrapper' => 'vragen-ai-systems-wrapper',
        'event' => 'change',
        'progress' => ['type' => 'throbber'],
      ],
    ];

    // AJAX wrapper for systems dropdown.
    $form['systems_container'] = [
      '#parents' => ['backend_config'],
      '#type' => 'container',
      '#attributes' => ['id' => 'vragen-ai-systems-wrapper'],
    ];

    $values = $form_state->getCompleteFormState()->getValues();

    $endpoint = $values['backend_config']['api_endpoint'] ?? $this->configuration['api_endpoint'] ?? '';
    $token = $values['backend_config']['api_token'] ?? $this->configuration['api_token'] ?? '';

    if (!empty($endpoint) && !empty($token)) {
      /** @var \Drupal\search_api_vragen_ai\Client\ClientFactory $clientFactory */
      $client = $this->clientFactory->createClient($endpoint, $token);

      try {
        $systems = $client->systems()->cachedAllByType('searchSystem', TRUE);
      }
      catch (\Exception $e) {
        $form['systems_container']['search_system'] = [
          '#markup' => '<div class="messages messages--warning">' . $this->t('Vragen.ai connection credentials are invalid.') . '</div>',
        ];

        return $form;
      }

      $options = [];
      foreach ($systems as $system) {
        $options[$system->uuid] = $system->tag ?? $system->name ?? $system->id;
      }

      $form['systems_container']['search_system'] = [
        '#type' => 'select',
        '#title' => $this->t('Search System'),
        '#description' => $this->t('Select the Vragen.ai search system to use when searching, leave empty to search without system.'),
        '#options' => $options,
        '#default_value' => $this->configuration['search_system'] ?? '',
        '#empty_option' => $this->t('- None -'),
      ];
    }
    else {
      $form['systems_container']['search_system'] = [
        '#markup' => '<div class="messages messages--warning">' . $this->t('Enter endpoint and token to load search systems.') . '</div>',
      ];
    }

    return $form;
  }

  /**
   * AJAX callback to rebuild the systems dropdown.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The updated systems container.
   */
  public static function updateSystemsDropdown(array &$form, FormStateInterface $form_state): array {
    // Search API sets a message on form rebuild, we need to clear it.
    \Drupal::messenger()->deleteAll();

    return $form['backend_config']['systems_container'];
  }

  /**
   * {@inheritdoc}
   */
  public function indexItems(IndexInterface $index, array $items): array {
    $indexed = [];

    foreach ($items as $id => $item) {
      try {
        $this->indexItem($index, $item);
        $indexed[] = $id;
      }
      catch (\Throwable $e) {
        $this->getLogger()->error($e->getMessage() . ' (Document: ' . $item->getId() . ')');
      }
    }

    return $indexed;
  }

  /**
   * Indexes the specified items.
   *
   * @param \Drupal\search_api\IndexInterface $index
   *   The index.
   * @param \Drupal\search_api\Item\ItemInterface $item
   *   The item to index.
   */
  protected function indexItem(IndexInterface $index, ItemInterface $item): void {
    $originalObject = $item->getOriginalObject();

    $datasource = $item->getDatasource();

    $document = $this->firstOrCreateDocument($item->getId());

    $published_date = date('Y-m-d');
    if ($originalObject instanceof EntityAdapter && $originalObject->getEntity() && $originalObject->getEntity()->hasField('created')) {
      $published_date = date('Y-m-d', (int) $originalObject->getEntity()->get('created')->getString());
    }

    $document->fill([
      'external_reference' => $item->getId(),
      'title' => $datasource->getItemLabel($originalObject) ?: 'Empty',
      'url' => $datasource->getItemUrl($originalObject)?->setAbsolute()->toString(),
      'mime_type' => 'text/html',
      'content' => $this->getIndexContentForItem($item),
      'meta_data' => array_merge([
        'author' => $originalObject instanceof EntityAdapter && $originalObject->getEntity() instanceof EntityOwnerInterface ? $originalObject->getEntity()->getOwner()->getDisplayName() : NULL,
        'entity_type' => $datasource->getEntityTypeId(),
        'bundle' => $datasource->getItemBundle($originalObject),
        'date' => $published_date,
      ], $this->getMetadataForItem($item)),
    ]);

    $event = new PostCreateIndexDocumentEvent($item, $document, $index);
    $this->eventDispatcher->dispatch($event);

    if ($event->shouldIndex()) {
      if ($document->isNew()) {
        $result = $this->getClient()->documents()->create($document);
      }
      else {
        $result = $this->getClient()->documents()->update($document);
      }
    }
    else {
      // If the event indicates the item should not be indexed, delete it if it
      // already exists.
      if (!$document->isNew() && !empty($document->getId())) {
        $result = $this->getClient()->documents()->delete($document->getId());
      }
    }

    if ($result?->getErrors()->isNotEmpty()) {
      $error = $result->getErrors()->first();
      throw new \Exception('Error when indexing item: ' . $error->getTitle() . ' (' . $error->getStatus() . ': ' . $error->getDetail() . ')');
    }
  }

  /**
   * {@inheritdoc}
   */
  public function deleteItems(IndexInterface $index, array $item_ids): void {
    foreach ($item_ids as $id) {
      $document = $this->firstOrCreateDocument($id);
      if ($document->isNew() || empty($document->getId())) {
        continue;
      }

      $this->getClient()->documents()->delete($document->getId());
    }
  }

  /**
   * {@inheritdoc}
   */
  public function supportsDataType($type): bool {
    if ($type === 'vragen_ai_metadata') {
      return TRUE;
    }
    return parent::supportsDataType($type);
  }

  /**
   * {@inheritdoc}
   */
  public function getSupportedFeatures(): array {
    return [
      'search_api_mlt',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function deleteAllIndexItems(IndexInterface $index, $datasource_id = NULL): void {
    // Not supported.
  }

  /**
   * {@inheritdoc}
   */
  public function search(QueryInterface $query): void {
    if ($query->hasTag('server_index_status')) {
      $documents = $this->getClient()->documents()->all();

      $results = $query->getResults();
      $results->setResultCount($documents->getMeta()->page->total ?? 0);
      return;
    }

    $index = $query->getIndex();
    $mlt_options = $query->getOption('search_api_mlt');
    $is_mlt = !empty($mlt_options);

    if (!$is_mlt) {
      $keys = $query->getKeys();
      if (empty($keys)) {
        return;
      }

      $searchPhrase = $this->getSearchPhrase($keys);
      if (empty($searchPhrase)) {
        return;
      }
    }
    else {
      assert(is_array($mlt_options));
      $mlt_id = $this->findMoreLikeThisId($mlt_options, $index);
      if (empty($mlt_id)) {
        return;
      }
    }

    $systems = $this->getClient()->systems()->cachedAllByType('searchSystem');
    $searchSystem = $systems[$this->configuration['search_system']] ?? NULL;

    try {
      $resultCollection = $is_mlt
        ? $this->getClient()->similar(
          $mlt_id,
          $query->getConditionGroup(),
          $query->getOption('offset'),
          $query->getOption('limit'),
        )
        : $this->getClient()->search(
          $searchPhrase,
          $query->getConditionGroup(),
          $query->getOption('offset'),
          $query->getOption('limit'),
          $searchSystem?->getId(),
        );
    }
    catch (\Throwable $e) {
      $this->getLogger()->error('Failed to search: ' . $e);
      return;
    }

    $results = $query->getResults();

    foreach ($resultCollection as $result) {
      // We can safely skip items that don't have an external reference.
      // This happens when items are not indexed using Search API.
      if (!$result->external_reference) {
        continue;
      }

      $item = $this->getFieldsHelper()->createItem($index, $result->external_reference);
      $item->setScore($result->relevance_score ?? 0);
      $results->addResultItem($item);
    }

    $results->setResultCount($resultCollection->total());
  }

  /**
   * Retrieves a proper external ID of the item to perform a similar search on.
   *
   * @param array $mlt_options
   *   The SAPI More Like This options.
   * @param \Drupal\search_api\IndexInterface $index
   *   The SAPI index to perform the search on.
   *
   * @return string|null
   *   The found ID or null.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  private function findMoreLikeThisId(array $mlt_options, IndexInterface $index): ?string {
    $id = $mlt_options['id'] ?? NULL;

    foreach ($index->getDatasources() as $datasource) {
      if ($entity_type_id = $datasource->getEntityTypeId()) {
        $entity = $this->entityTypeManager
          ->getStorage($entity_type_id)
          ->load($mlt_options['id']);

        if ($entity) {
          $id = SearchApiUtility::createCombinedId(
            $datasource->getPluginId(),
            $datasource->getItemId($entity->getTypedData()),
          );
          break;
        }
      }
    }

    if (empty($id)) {
      $this->getLogger()->warning('More Like This requested but no valid document ID could be resolved.');
      return NULL;
    }

    return $id;
  }

  /**
   * {@inheritdoc}
   */
  public function isAvailable(): bool {
    try {
      $response = $this->getClient()->systems()->all();

      return $response instanceof CollectionDocumentInterface;
    }
    catch (\Throwable $e) {
      $this->getLogger()->error('Vragen AI is not available: ' . $e->getMessage());
      return FALSE;
    }
  }

  /**
   * Get first document or create and return one.
   *
   * @param string $externalReference
   *   The external reference ID.
   *
   * @return \Drupal\search_api_vragen_ai\Client\DocumentItem
   *   The document.
   */
  protected function firstOrCreateDocument(string $externalReference): DocumentItem {
    $document = $this->getClient()->documents()->findByExternalReference($externalReference);

    if ($document instanceof DocumentItem) {
      return $document;
    }

    return $this->getClient()->documents()->newByExternalReference($externalReference);
  }

  /**
   * Get the index content for the item.
   *
   * @param \Drupal\search_api\Item\ItemInterface $item
   *   The item to get the content for.
   *
   * @return string
   *   The index content.
   */
  protected function getIndexContentForItem(ItemInterface $item): string {
    $contentFields = [];
    foreach ($item->getFields() as $field) {
      $fieldValues = match($field->getType()) {
        'text', 'string', 'integer' => $field->getValues(),
        'boolean' => array_map(fn ($value) => sprintf('%s: %s', $field->getLabel(), $value ? 'true' : 'false'), $field->getValues()),
        default => NULL,
      };

      if (empty($fieldValues)) {
        continue;
      }

      $fieldValues = array_map(fn ($value) => $this->hasBlockLevelElements($value) ? $value : sprintf('<p>%s</p>', $value), $fieldValues);
      $contentFields[] = implode("\n", $fieldValues);
    }

    return sprintf('<!doctype html><html><body>%s</body></html>', implode("\n", $contentFields));
  }

  /**
   * Get the metadata for the item.
   *
   * @param \Drupal\search_api\Item\ItemInterface $item
   *   The item to get the metadata for.
   *
   * @return array<string, mixed>
   *   The metadata.
   */
  protected function getMetadataForItem(ItemInterface $item): array {
    $metadata = [];
    foreach ($item->getFields() as $field) {
      if ($field->getType() !== 'vragen_ai_metadata') {
        continue;
      }

      $fieldValues = $field->getValues();
      if (empty($fieldValues)) {
        continue;
      }

      if (count($fieldValues) > 1) {
        $metadata[$field->getFieldIdentifier()] = $fieldValues;
      }
      else {
        $metadata[$field->getFieldIdentifier()] = $fieldValues[0];
      }
    }

    return $metadata;
  }

  /**
   * Checks if the content has block level elements.
   *
   * @param string $content
   *   The content to check.
   *
   * @return bool
   *   TRUE if the content has block level elements, FALSE otherwise.
   */
  protected function hasBlockLevelElements(string $content): bool {
    return strip_tags($content) !== strip_tags($content, '<a><b><i><strong><em><span><button><br><code><abbr><sup><sub><mark><small><time>');
  }

  /**
   * Returns the search phrase from the keys.
   *
   * @param array $keys
   *   The search keys to build the search phrase from.
   *
   * @return string
   *   The search phrase.
   */
  protected function getSearchPhrase(array $keys): string {
    $words = [];

    foreach ($keys as $i => $key) {
      if (!Element::child($i)) {
        continue;
      }
      if (is_scalar($key)) {
        $words[] = $key;
      }
    }

    return implode(' ', $words);
  }

  /**
   * {@inheritdoc}
   */
  public function __sleep(): array {
    $properties = array_flip(parent::__sleep());

    unset($properties['client']);

    return array_keys($properties);
  }

}
