<?php

declare(strict_types=1);

namespace Drupal\search_api_yext\Plugin\search_api\backend;

use Drupal\search_api\Plugin\search_api\tracker\Basic;
use Drupal\Component\Utility\Html;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Plugin\PluginFormInterface;
use Drupal\key\KeyRepositoryInterface;
use Drupal\search_api\Backend\BackendPluginBase;
use Drupal\search_api\IndexInterface;
use Drupal\search_api\Item\FieldInterface;
use Drupal\search_api\Item\ItemInterface;
use Drupal\search_api\Plugin\PluginFormTrait;
use Drupal\search_api\Query\QueryInterface;
use Drupal\search_api\Query\ResultSetInterface;
use Drupal\search_api_yext\YextApiClient;
use Drupal\search_api\Utility\Utility;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Yext SearchApiBackend definition.
 *
 * @SearchApiBackend(
 *   id = "search_api_yext",
 *   label = @Translation("Yext"),
 *   description = @Translation("Index items using Yext Push Connectors.")
 * )
 */
class SearchApiYextBackend extends BackendPluginBase implements PluginFormInterface {

  use PluginFormTrait;

  /**
   * The logger to use for logging messages.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected LanguageManagerInterface $languageManager;

  /**
   * The Config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * Yext API client service.
   *
   * @var \Drupal\search_api_yext\YextApiClient
   */
  protected YextApiClient $yextApiClient;

  /**
   * The key repository service.
   *
   * @var \Drupal\key\KeyRepositoryInterface
   */
  protected KeyRepositoryInterface $keyRepository;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected Connection $database;

  /**
   * The entity field manager.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected EntityFieldManagerInterface $entityFieldManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);

    $instance->languageManager = $container->get('language_manager');
    $instance->configFactory = $container->get('config.factory');
    $instance->moduleHandler = $container->get('module_handler');
    $instance->logger = $container->get('logger.channel.search_api_yext');
    $instance->keyRepository = $container->get('key.repository');
    $instance->database = $container->get('database');
    $instance->entityFieldManager = $container->get('entity_field.manager');

    // Create YextApiClient with configured base URL.
    $api_base_url = $configuration['api_base_url'] ?? 'https://api.yext.com/v2';
    $instance->yextApiClient = new YextApiClient(
      $container->get('entity_type.manager'),
      $container->get('language_manager'),
      $container->get('database'),
      $container->get('module_handler'),
      $container->get('http_client'),
      $container->get('logger.channel.search_api_yext'),
      $api_base_url
    );

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'account_id' => '',
      'api_key' => '',
      'api_base_url' => 'https://api.yext.com/v2',
      'disable_truncate' => FALSE,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form['help'] = [
      '#markup' => '<p>' . $this->t('The Account ID and API key can be found in your Yext account settings at <a href="@link" target="blank">@link</a>.', ['@link' => 'https://www.yext.com/admin/account']) . '</p>',
    ];

    $form['account_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Account ID'),
      '#description' => $this->t('The Account ID from your Yext subscription.'),
      '#default_value' => $this->getAccountId(),
      '#required' => TRUE,
      '#size' => 60,
      '#maxlength' => 128,
    ];

    $form['api_key'] = [
      '#type' => 'key_select',
      '#title' => $this->t('API Key'),
      '#description' => $this->t('The API key from your Yext connector app. Used for both indexing and deletion operations.'),
      '#default_value' => $this->configuration['api_key'],
      '#required' => TRUE,
      '#empty_option' => $this->t('- Select a key -'),
    ];

    $form['api_base_url'] = [
      '#type' => 'url',
      '#title' => $this->t('API Base URL'),
      '#description' => $this->t('The base URL for the Yext API. Default is https://api.yext.com/v2'),
      '#default_value' => $this->configuration['api_base_url'],
      '#required' => TRUE,
    ];

    $form['disable_truncate'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Disable truncation'),
      '#description' => $this->t('If checked, fields of type text and string will not be truncated at 10000 characters. It will be site owner or developer responsibility to limit the characters.'),
      '#default_value' => $this->configuration['disable_truncate'],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function viewSettings(): array {
    $info = [];

    // Account ID.
    $info[] = [
      'label' => $this->t('Account ID'),
      'info' => $this->getAccountId(),
    ];

    // API Key.
    $key_id = $this->configuration['api_key'];
    $key_info = $key_id;
    if (!empty($key_id)) {
      $key = $this->keyRepository->getKey($key_id);
      if ($key) {
        $key_info = $key->label() . ' (' . $key_id . ')';
      }
      else {
        $key_info = $this->t('Key not found: @key_id', ['@key_id' => $key_id]);
      }
    }
    $info[] = [
      'label' => $this->t('API Key'),
      'info' => $key_info,
    ];

    // Connection status.
    $is_connected = $this->yextApiClient->testConnection($this->getAccountId(), $this->getApiKey());
    $message = $is_connected
      ? $this->t('The Yext backend could be reached.')
      : $this->t('The Yext backend could not be reached.');
    $info[] = [
      'label' => $this->t('Connection status'),
      'info' => $message,
      'status' => $is_connected ? 'success' : 'error',
    ];

    return $info;
  }

  /**
   * {@inheritdoc}
   *
   * @throws \Drupal\search_api\SearchApiException
   */
  public function removeIndex($index): void {
    // Only delete the index's data if the index isn't read-only.
    if (!is_object($index) || empty($index->get('read_only'))) {
      $this->deleteAllIndexItems($index);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function indexItems(IndexInterface $index, array $items): array {
    $entities = [];

    /** @var \Drupal\search_api\Item\ItemInterface $item */
    foreach ($items as $id => $item) {
      $entities[$id] = $this->prepareItem($index, $item);
    }

    if (count($entities) > 0) {
      $itemsToIndex = [];

      if ($this->languageManager->isMultilingual()) {
        foreach ($entities as $item) {
          $itemsToIndex[$item['search_api_language']][] = $item;
        }
      }
      else {
        $itemsToIndex[''] = $entities;
      }

      foreach ($itemsToIndex as $items) {
        try {
          $connector_name = $index->getOption('yext_connector_name', '');
          $this->yextApiClient->indexEntities($this->getAccountId(), $this->getApiKey(), $items, $connector_name);
        }
        catch (\Exception $e) {
          $this->logger->warning(Html::escape($e->getMessage()));
        }
      }
    }

    return array_keys($entities);
  }

  /**
   * Prepares a single item for indexing.
   *
   * @param \Drupal\search_api\IndexInterface $index
   *   Index.
   * @param \Drupal\search_api\Item\ItemInterface $item
   *   The item to index.
   *
   * @return array
   *   Item to index.
   */
  protected function prepareItem(IndexInterface $index, ItemInterface $item): array {
    $item_id = $item->getId();
    $yext_entity_id = $this->yextApiClient->convertToYextEntityId($item_id);
    $item_to_index = ['entityId' => $yext_entity_id];
    $entity_id_field = $index->getOption('entity_id_field');

    // Change entityId if some other field is used in the config.
    if ($entity_id_field) {
      $entity = $item->getOriginalObject()->getValue();
      if ($entity instanceof ContentEntityInterface) {
        // Use the value of the field set in entity_id_field config as entityId.
        $entity_id = $entity->hasField($entity_id_field) ? $entity->get($entity_id_field)->getString() : '';
        if ($entity_id) {
          $item_to_index['entityId'] = $entity_id;
        }
      }
    }

    $item_fields = $item->getFields();
    $item_fields += $this->getSpecialFields($index, $item);

    /** @var \Drupal\search_api\Item\FieldInterface $field */
    foreach ($item_fields as $field) {
      $type = $field->getType();
      $values = NULL;
      $field_values = $field->getValues();
      if (empty($field_values)) {
        continue;
      }
      foreach ($field_values as $field_value) {
        switch ($type) {
          case 'uri':
            $field_value .= '';
            if (mb_strlen($field_value) > 10000) {
              $field_value = mb_substr(trim($field_value), 0, 10000);
            }
            $values[] = $field_value;
            break;

          case 'text':
          case 'string':
            $field_value .= '';
            if (empty($this->configuration['disable_truncate']) && mb_strlen($field_value) > 10000) {
              $field_value = mb_substr(trim($field_value), 0, 10000);
            }
            $values[] = $field_value;
            break;

          case 'integer':
          case 'duration':
          case 'decimal':
            $values[] = 0 + $field_value;
            break;

          case 'boolean':
            $values[] = $field_value ? TRUE : FALSE;
            break;

          case 'date':
            if (is_numeric($field_value) || !$field_value) {
              $values[] = 0 + $field_value;
              break;
            }
            $values[] = strtotime($field_value);
            break;

          default:
            $values[] = $field_value;
        }
      }
      // Preserve array structure for multi-value fields even when only a
      // single value is present. Collapsing to a scalar causes downstream
      // systems to interpret comma-containing values as separate entries. Only
      // collapse if this index field is configured as single-value and we have
      // exactly one value.
      if (is_array($values) && count($values) <= 1) {
        $cardinality = $this->getFieldCardinality($index, $field);
        if ($cardinality === 1) {
          $values = reset($values);
        }
      }
      $item_to_index[$field->getFieldIdentifier()] = $values;
    }

    return $item_to_index;
  }

  /**
   * {@inheritdoc}
   */
  public function deleteItems(IndexInterface $index, array $ids): void {
    // When using custom field for entity id, we handle the deletion of
    // entities in separate code.
    if ($index->getOption('entity_id_field')) {
      return;
    }

    // Convert Search API item IDs to Yext entity IDs.
    $yext_entity_ids = [];
    foreach ($ids as $item_id) {
      $yext_entity_ids[] = $this->yextApiClient->convertToYextEntityId($item_id);
    }

    try {
      $connector_name = $index->getOption('yext_connector_name', '');
      $this->yextApiClient->deleteEntities($this->getAccountId(), $this->getApiKey(), $connector_name, $yext_entity_ids);
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to delete items from Yext, Error: @message', [
        '@message' => $e->getMessage(),
      ]);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function deleteAllIndexItems(?IndexInterface $index = NULL, $datasource_id = NULL): void {
    if (empty($index)) {
      return;
    }

    // Get all indexed item IDs using the tracker's database table.
    $tracker = $index->getTrackerInstance();
    if ($tracker instanceof Basic) {
      $query = $this->database->select('search_api_item', 'sai')
        ->fields('sai', ['item_id'])
        ->condition('sai.index_id', $index->id())
      // STATUS_INDEXED.
        ->condition('sai.status', 1);

      if ($datasource_id) {
        $query->condition('sai.datasource', $datasource_id);
      }

      $item_ids = $query->execute()->fetchCol();

      if (!empty($item_ids)) {
        // Delete items in batches to avoid memory issues.
        $batch_size = 100;
        $batches = array_chunk($item_ids, $batch_size);

        foreach ($batches as $batch) {
          try {
            $this->deleteItems($index, $batch);
            $this->logger->info('Deleted batch of @count items from Yext index.', [
              '@count' => count($batch),
            ]);
          }
          catch (\Exception $e) {
            $this->logger->error('Failed to delete batch from Yext, Error: @message', [
              '@message' => $e->getMessage(),
            ]);
          }
        }
      }
    }
    else {
      // For other tracker types, log that clearing all items is not supported.
      $this->logger->notice('Clearing all index items is only supported for Basic tracker. Items must be deleted individually.');
    }
  }

  /**
   * {@inheritdoc}
   */
  public function search(QueryInterface $query): ResultSetInterface {
    // Note: Yext search is typically handled by frontend JavaScript components.
    // This backend is primarily for indexing content, not searching.
    $results = $query->getResults();
    return $results;
  }

  /**
   * Get the Account ID (provided by Yext).
   */
  protected function getAccountId(): string {
    return $this->configuration['account_id'];
  }

  /**
   * Get the API key (provided by Yext).
   */
  protected function getApiKey(): string {
    $key_id = $this->configuration['api_key'];
    if (empty($key_id)) {
      return '';
    }

    $key = $this->keyRepository->getKey($key_id);
    if (!$key) {
      $this->logger->error('Key with ID "@key_id" not found.', ['@key_id' => $key_id]);
      return '';
    }

    try {
      return $key->getKeyValue();
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to retrieve key value: @message', ['@message' => $e->getMessage()]);
      return '';
    }
  }

  /**
   * Gets the Drupal field cardinality for the given Search API field.
   */
  protected function getFieldCardinality(IndexInterface $index, FieldInterface $field): ?int {
    $property_path = $field->getPropertyPath();
    [$base_property] = Utility::splitPropertyPath($property_path, FALSE);

    if (!$base_property) {
      return 1;
    }

    $datasource_id = $field->getDatasourceId();
    $entity_type_id = NULL;
    if (is_string($datasource_id) && str_starts_with($datasource_id, 'entity:')) {
      [, $entity_type_id] = explode(':', $datasource_id, 2);
    }

    if (!$entity_type_id) {
      return 1;
    }

    $field_storages = $this->entityFieldManager->getFieldStorageDefinitions($entity_type_id);

    if (!isset($field_storages[$base_property])) {
      return 1;
    }

    return $field_storages[$base_property]->getCardinality();
  }

}
