<?php

declare(strict_types=1);

namespace Drupal\search_api_yext;

use Drupal\Component\Utility\UrlHelper;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use GuzzleHttp\Client;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\GuzzleException;
use GuzzleHttp\Exception\RequestException;
use Psr\Log\LoggerInterface;

/**
 * Yext API client service.
 */
class YextApiClient {

  /**
   * The Yext API base URL.
   *
   * @var string
   */
  protected string $apiBaseUrl;

  /**
   * Constructs a new class instance.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Language\LanguageManagerInterface $languageManager
   *   The language manager.
   * @param \Drupal\Core\Database\Connection $connection
   *   Database connection object.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler.
   * @param \GuzzleHttp\ClientInterface $httpClient
   *   The HTTP client.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger.
   * @param string $apiBaseUrl
   *   The API base URL.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected LanguageManagerInterface $languageManager,
    protected Connection $connection,
    protected ModuleHandlerInterface $moduleHandler,
    protected ClientInterface $httpClient,
    protected LoggerInterface $logger,
    string $apiBaseUrl = 'https://api.yext.com/v2',
  ) {
    $this->apiBaseUrl = $apiBaseUrl;
  }

  /**
   * Convert Search API item ID to Yext entity ID.
   *
   * Converts "entity:node/39686:en" to "node-39686-en".
   *
   * @param string $search_api_item_id
   *   Search API item ID.
   *
   * @return string
   *   Yext entity ID.
   */
  public function convertToYextEntityId(string $search_api_item_id): string {
    // Remove "entity:" prefix and replace forward slashes and colons with
    // dashes.
    $yext_id = str_replace('entity:', '', $search_api_item_id);
    $yext_id = str_replace(['/', ':'], '-', $yext_id);
    return $yext_id;
  }

  /**
   * Convert Yext entity ID back to Search API item ID.
   *
   * Converts "node-39686-en" to "entity:node/39686:en".
   *
   * @param string $yext_entity_id
   *   Yext entity ID.
   *
   * @return string
   *   Search API item ID.
   */
  public function convertFromYextEntityId(string $yext_entity_id): string {
    // Split on dashes and reconstruct Search API format.
    $parts = explode('-', $yext_entity_id);
    if (count($parts) >= 3) {
      $entity_type = $parts[0];
      $entity_id = $parts[1];
      $language = $parts[2];
      return "entity:{$entity_type}/{$entity_id}:{$language}";
    }
    // Fallback if format doesn't match expected pattern.
    return "entity:{$yext_entity_id}";
  }

  /**
   * Get HTTP client with proper typing.
   *
   * @return \GuzzleHttp\Client
   *   The HTTP client with get/post methods.
   */
  private function getHttpClient(): Client {
    if (!$this->httpClient instanceof Client) {
      throw new \RuntimeException('HTTP client must be an instance of GuzzleHttp\Client');
    }
    return $this->httpClient;
  }

  /**
   * Build and sanitize Yext API URL using Drupal methods.
   *
   * @param string $account_id
   *   The account ID.
   * @param string $connector_name
   *   The connector name (optional).
   * @param string $endpoint
   *   The API endpoint (e.g., 'entities', 'pushData').
   *
   * @return string
   *   The sanitized URL.
   *
   * @throws \InvalidArgumentException
   *   If parameters are empty or invalid.
   */
  private function buildApiUrl(string $account_id, string $connector_name = '', string $endpoint = 'entities'): string {
    // Validate required parameters.
    if (empty($account_id)) {
      throw new \InvalidArgumentException('Account ID cannot be empty.');
    }
    if (empty($endpoint)) {
      throw new \InvalidArgumentException('Endpoint cannot be empty.');
    }
    // Sanitize and validate the base URL.
    $base_url = trim($this->apiBaseUrl);
    if (empty($base_url)) {
      throw new \InvalidArgumentException('API base URL cannot be empty.');
    }

    // Ensure the base URL is valid and uses HTTPS.
    if (!UrlHelper::isValid($base_url, TRUE)) {
      throw new \InvalidArgumentException('API base URL is not a valid URL.');
    }

    // Remove trailing slash from base URL to prevent double slashes.
    $base_url = rtrim($base_url, '/');

    // Use Drupal's UrlHelper to encode URL components safely.
    $account_id = UrlHelper::encodePath($account_id);
    $endpoint = UrlHelper::encodePath($endpoint);

    if (!empty($connector_name)) {
      $connector_name = UrlHelper::encodePath($connector_name);
      return $base_url . '/accounts/' . $account_id . '/connectors/' . $connector_name . '/' . $endpoint;
    }
    return $base_url . '/accounts/' . $account_id . '/' . $endpoint;
  }

  /**
   * Test connection to Yext API.
   *
   * @param string $account_id
   *   Account ID.
   * @param string $api_key
   *   API key.
   *
   * @return bool
   *   TRUE if connection is successful.
   */
  public function testConnection(string $account_id, string $api_key): bool {
    try {
      $client = $this->getHttpClient();
      $url = $this->buildApiUrl($account_id, '', 'entities');
      $response = $client->get($url, [
        'query' => [
          'api_key' => $api_key,
          'v' => date('Ymd'),
          'limit' => 1,
        ],
        'timeout' => 10,
      ]);

      return $response->getStatusCode() === 200;
    }
    catch (RequestException $e) {
      $this->logger->error('Yext API connection test failed (HTTP error): @message', [
        '@message' => $e->getMessage(),
      ]);
      return FALSE;
    }
    catch (GuzzleException $e) {
      $this->logger->error('Yext API connection test failed (network error): @message', [
        '@message' => $e->getMessage(),
      ]);
      return FALSE;
    }
  }

  /**
   * Index entities to Yext.
   *
   * @param string $account_id
   *   Account ID.
   * @param string $api_key
   *   API key.
   * @param array $entities
   *   Array of entities to index.
   * @param string $connector_name
   *   Yext connector name.
   *
   * @return array
   *   Response from Yext API.
   *
   * @throws \Exception
   */
  public function indexEntities(string $account_id, string $api_key, array $entities, string $connector_name = 'drupalArticles'): array {
    $results = [];

    // Create new entity.
    try {
      $client = $this->getHttpClient();
      $url = $this->buildApiUrl($account_id, $connector_name, 'pushData');
      $response = $client->post($url, [
        'query' => [
          'api_key' => $api_key,
          'v' => date('Ymd'),
        ],
        'json' => $entities,
        'headers' => [
          'Content-Type' => 'application/json',
        ],
      ]);
      $results[] = [
        'status_code' => $response->getStatusCode(),
        'response' => json_decode($response->getBody()->getContents(), TRUE),
      ];
    }
    catch (RequestException $e) {
      $this->logger->error('Failed to index entity @entity_id to Yext (HTTP error): @message', [
        '@entity_id' => $entities[0]['entityId'] ?? 'unknown',
        '@message' => $e->getMessage(),
      ]);
      throw new \Exception('Failed to index entity to Yext: ' . $e->getMessage());
    }
    catch (GuzzleException $e) {
      $this->logger->error('Failed to index entity @entity_id to Yext (network error): @message', [
        '@entity_id' => $entities[0]['entityId'] ?? 'unknown',
        '@message' => $e->getMessage(),
      ]);
      throw new \Exception('Failed to index entity to Yext: ' . $e->getMessage());
    }
    return $results;
  }

  /**
   * Delete entities from Yext using connector API with runMode=DELETION.
   *
   * @param string $account_id
   *   Account ID.
   * @param string $api_key
   *   API key.
   * @param string $connector_name
   *   Yext connector name.
   * @param array $entity_ids
   *   Array of entity IDs to delete.
   *
   * @return array
   *   Response from Yext API.
   *
   * @throws \Exception
   */
  public function deleteEntities(string $account_id, string $api_key, string $connector_name, array $entity_ids): array {
    $results = [];

    // Prepare entities for deletion - just need entityId for each.
    $entities_to_delete = [];
    foreach ($entity_ids as $entity_id) {
      $entities_to_delete[] = ['entityId' => $entity_id];
    }

    try {
      $client = $this->getHttpClient();
      $url = $this->buildApiUrl($account_id, $connector_name, 'pushData');
      $response = $client->post($url, [
        'query' => [
          'api_key' => $api_key,
          'v' => date('Ymd'),
          'runMode' => 'DELETION',
        ],
        'json' => $entities_to_delete,
        'headers' => [
          'Content-Type' => 'application/json',
        ],
      ]);

      $results[] = [
        'entity_ids' => $entity_ids,
        'status_code' => $response->getStatusCode(),
        'response' => json_decode($response->getBody()->getContents(), TRUE),
      ];

      $this->logger->info('Successfully requested deletion of @count entities from Yext', [
        '@count' => count($entity_ids),
      ]);
    }
    catch (RequestException $e) {
      $this->logger->error('Failed to delete entities from Yext (HTTP error): @message', [
        '@message' => $e->getMessage(),
      ]);
      throw new \Exception('Failed to delete entities from Yext: ' . $e->getMessage());
    }
    catch (GuzzleException $e) {
      $this->logger->error('Failed to delete entities from Yext (network error): @message', [
        '@message' => $e->getMessage(),
      ]);
      throw new \Exception('Failed to delete entities from Yext: ' . $e->getMessage());
    }

    return $results;
  }

}
