<?php

declare(strict_types=1);

namespace Drupal\Tests\search_api_yext\Kernel;

use Drupal\entity_test\Entity\EntityTest;
use Drupal\KernelTests\KernelTestBase;
use Drupal\search_api\Entity\Index;
use Drupal\search_api\Entity\Server;
use Drupal\search_api\IndexInterface;
use Drupal\search_api\Item\Item;
use Drupal\search_api\Item\ItemInterface;
use Drupal\search_api\Plugin\search_api\data_type\value\TextValue;
use Drupal\search_api\ServerInterface;
use Drupal\search_api_yext\Plugin\search_api\backend\SearchApiYextBackend;
use GuzzleHttp\Client;
use GuzzleHttp\Handler\MockHandler;
use GuzzleHttp\HandlerStack;
use GuzzleHttp\Psr7\Response;

/**
 * Tests for the Yext Search API backend.
 *
 * @group search_api_yext
 */
class SearchApiYextBackendTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'search_api',
    'search_api_test',
    'search_api_yext',
    'user',
    'system',
    'entity_test',
    'key',
  ];

  /**
   * A search server instance.
   *
   * @var \Drupal\search_api\ServerInterface
   */
  protected ServerInterface $server;

  /**
   * A search index instance.
   *
   * @var \Drupal\search_api\IndexInterface
   */
  protected IndexInterface $index;

  /**
   * The backend plugin instance.
   *
   * @var \Drupal\search_api_yext\Plugin\search_api\backend\SearchApiYextBackend
   */
  protected SearchApiYextBackend $backend;

  /**
   * Mock HTTP handler for testing API calls.
   *
   * @var \GuzzleHttp\Handler\MockHandler
   */
  protected MockHandler $mockHandler;

  /**
   * Test entity for tests that need it.
   *
   * @var \Drupal\entity_test\Entity\EntityTest
   */
  protected EntityTest $testEntity;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('entity_test');
    $this->installEntitySchema('search_api_task');
    $this->installSchema('search_api', ['search_api_item']);

    // Set tracking page size so tracking will work properly.
    \Drupal::configFactory()
      ->getEditable('search_api.settings')
      ->set('tracking_page_size', 100)
      ->save();

    // Create a test entity for tests that need it.
    $this->testEntity = \Drupal::entityTypeManager()
      ->getStorage('entity_test')
      ->create([
        'id' => 1,
        'name' => 'Test Entity',
        'langcode' => 'en',
      ]);
    $this->testEntity->save();

    // Create a mock HTTP handler for Guzzle.
    $this->mockHandler = new MockHandler();
    $handlerStack = HandlerStack::create($this->mockHandler);
    $client = new Client(['handler' => $handlerStack]);

    // Override the HTTP client service.
    $this->container->set('http_client', $client);

    // Create server.
    $this->server = Server::create([
      'name' => 'Test Yext Server',
      'id' => 'test_yext_server',
      'backend' => 'search_api_yext',
      'backend_config' => [
        'account_id' => 'test_account',
        'api_key' => 'test_api_key',
        'disable_truncate' => FALSE,
      ],
    ]);
    $this->server->save();

    // Create index.
    $this->index = Index::create([
      'name' => 'Test Yext Index',
      'id' => 'test_yext_index',
      'status' => TRUE,
      'server' => $this->server->id(),
      'datasource_settings' => [
        'entity:entity_test' => [],
      ],
      'tracker_settings' => [
        'default' => [
          'indexing_order' => 'fifo',
        ],
      ],
      'options' => [
        'cron_limit' => 50,
        'index_directly' => TRUE,
      ],
    ]);
    $this->index->save();

    // Add the yext_connector_name option after saving to avoid schema
    // validation.
    $this->index->setOption('yext_connector_name', 'test_connector');

    $this->backend = $this->server->getBackend();
  }

  /**
   * Tests backend plugin instantiation.
   */
  public function testBackendInstantiation(): void {
    $this->assertInstanceOf(SearchApiYextBackend::class, $this->backend);
    $this->assertEquals('search_api_yext', $this->backend->getPluginId());
  }

  /**
   * Tests default configuration.
   */
  public function testDefaultConfiguration(): void {
    $config = $this->backend->defaultConfiguration();

    $expected = [
      'account_id' => '',
      'api_key' => '',
      'disable_truncate' => FALSE,
    ];

    $this->assertEquals($expected, $config);
  }

  /**
   * Tests configuration form building.
   */
  public function testBuildConfigurationForm(): void {
    $form = [];
    /** @var \Drupal\Core\Form\FormStateInterface $form_state */
    $form_state = $this->createMock('\Drupal\Core\Form\FormStateInterface');

    $form = $this->backend->buildConfigurationForm($form, $form_state);

    $this->assertArrayHasKey('account_id', $form);
    $this->assertArrayHasKey('api_key', $form);
    $this->assertArrayHasKey('disable_truncate', $form);

    $this->assertEquals('textfield', $form['account_id']['#type']);
    $this->assertEquals('key_select', $form['api_key']['#type']);
    $this->assertEquals('checkbox', $form['disable_truncate']['#type']);
  }

  /**
   * Tests view settings.
   */
  public function testViewSettings(): void {
    // Mock successful connection test.
    $this->mockHandler->append(new Response(200, [], '{"entities": []}'));

    $settings = $this->backend->viewSettings();

    $this->assertIsArray($settings);
    $this->assertCount(3, $settings);

    // Check that all expected settings are present.
    $labels = array_map(function ($label) {
      return (string) $label;
    }, array_column($settings, 'label'));

    $this->assertContains('Account ID', $labels);
    $this->assertContains('API Key', $labels);
    $this->assertContains('Connection status', $labels);

    // Check connection status shows success.
    $connection_status = array_filter($settings, function ($setting) {
      return $setting['label'] == 'Connection status';
    });
    $this->assertEquals('success', reset($connection_status)['status']);
  }

  /**
   * Tests item preparation for indexing.
   */
  public function testPrepareItem(): void {
    // Create a test item.
    $item = $this->createTestItem();

    // Use reflection to access protected method.
    $reflection = new \ReflectionClass($this->backend);
    $method = $reflection->getMethod('prepareItem');
    $method->setAccessible(TRUE);

    $prepared_item = $method->invoke($this->backend, $this->index, $item);

    $this->assertIsArray($prepared_item);
    $this->assertArrayHasKey('entityId', $prepared_item);
    $this->assertEquals('entity_test-1-en', $prepared_item['entityId']);

    // Check field processing.
    $this->assertArrayHasKey('title', $prepared_item);
    $this->assertEquals('Test Title', $prepared_item['title']);
  }

  /**
   * Tests successful item indexing.
   */
  public function testIndexItems(): void {
    // Mock entity existence check (entity doesn't exist - 404)
    $this->mockHandler->append(new Response(404));

    // Mock successful entity creation.
    $this->mockHandler->append(new Response(201, [], '{"id": "entity:entity_test/1:en"}'));

    $item = $this->createTestItem();
    $items = ['entity:entity_test/1:en' => $item];

    $indexed_ids = $this->backend->indexItems($this->index, $items);

    $this->assertEquals(['entity:entity_test/1:en'], $indexed_ids);
  }

  /**
   * Tests item indexing with existing entity (update).
   */
  public function testIndexItemsUpdate(): void {
    // Mock entity existence check (entity exists - 200)
    $this->mockHandler->append(new Response(200, [], '{"id": "entity:entity_test/1:en"}'));

    // Mock successful entity update.
    $this->mockHandler->append(new Response(200, [], '{"id": "entity:entity_test/1:en"}'));

    $item = $this->createTestItem();
    $items = ['entity:entity_test/1:en' => $item];

    $indexed_ids = $this->backend->indexItems($this->index, $items);

    $this->assertEquals(['entity:entity_test/1:en'], $indexed_ids);
  }

  /**
   * Tests item deletion.
   */
  public function testDeleteItems(): void {
    // Mock successful deletion.
    $this->mockHandler->append(new Response(200, [], '{"status": "deleted"}'));

    $ids = ['entity:entity_test/1:en'];

    // This should not throw any exceptions.
    $this->backend->deleteItems($this->index, $ids);

    $this->addToAssertionCount(1);
  }

  /**
   * Tests delete all index items.
   */
  public function testDeleteAllIndexItems(): void {
    // This should not throw any exceptions (logs a notice about not being
    // supported).
    $this->backend->deleteAllIndexItems($this->index);

    $this->addToAssertionCount(1);
  }

  /**
   * Tests search method.
   */
  public function testSearch(): void {
    $query = $this->index->query();
    $query->keys('test search');

    $results = $this->backend->search($query);

    // Search should return empty results as it's handled by frontend
    // components.
    $this->assertEquals(0, $results->getResultCount());
    $this->assertEquals([], $results->getResultItems());
  }

  /**
   * Tests multilingual support.
   */
  public function testMultilingualSupport(): void {
    // Enable language module for this test.
    $this->enableModules(['language']);

    $item = $this->createTestItem();

    // Create language field properly.
    $fields_helper = \Drupal::getContainer()->get('search_api.fields_helper');
    $language_field = $fields_helper->createField($this->index, 'search_api_language', ['type' => 'string']);
    $language_field->addValue('en');
    $item->setField('search_api_language', $language_field);

    $items = ['entity:entity_test/1:en' => $item];

    // Mock API calls.
    $this->mockHandler->append(new Response(404));
    $this->mockHandler->append(new Response(201, [], '{"id": "entity:entity_test/1:en"}'));

    $indexed_ids = $this->backend->indexItems($this->index, $items);

    $this->assertEquals(['entity:entity_test/1:en'], $indexed_ids);
  }

  /**
   * Creates a test search API item.
   *
   * @return \Drupal\search_api\Item\ItemInterface
   *   A test item.
   */
  protected function createTestItem(): ItemInterface {
    $item = new Item($this->index, 'entity:entity_test/1:en');

    // Add some test fields using the fields helper service.
    $fields_helper = \Drupal::getContainer()->get('search_api.fields_helper');

    $title_field = $fields_helper->createField($this->index, 'title', ['type' => 'string']);
    $title_field->addValue(new TextValue('Test Title'));
    $item->setField('title', $title_field);

    $body_field = $fields_helper->createField($this->index, 'body', ['type' => 'text']);
    $body_field->addValue(new TextValue('This is a test body content for the search item.'));
    $item->setField('body', $body_field);

    $integer_field = $fields_helper->createField($this->index, 'test_integer', ['type' => 'integer']);
    $integer_field->addValue(42);
    $item->setField('test_integer', $integer_field);

    $boolean_field = $fields_helper->createField($this->index, 'test_boolean', ['type' => 'boolean']);
    $boolean_field->addValue(TRUE);
    $item->setField('test_boolean', $boolean_field);

    return $item;
  }

}
