<?php

declare(strict_types=1);

namespace Drupal\Tests\search_api_yext\Unit;

use GuzzleHttp\Psr7\Request;
use GuzzleHttp\Exception\ConnectException;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\search_api_yext\YextApiClient;
use Drupal\Tests\UnitTestCase;
use GuzzleHttp\Client;
use GuzzleHttp\Psr7\Response;
use Psr\Log\LoggerInterface;

/**
 * Simple tests for YextApiClient.
 *
 * @coversDefaultClass \Drupal\search_api_yext\YextApiClient
 * @group search_api_yext
 */
class SimpleYextApiClientTest extends UnitTestCase {

  /**
   * The YextApiClient instance.
   *
   * @var \Drupal\search_api_yext\YextApiClient
   */
  protected YextApiClient $yextApiClient;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    /** @var \Drupal\Core\Entity\EntityTypeManagerInterface|\PHPUnit\Framework\MockObject\MockObject $entity_type_manager */
    $entity_type_manager = $this->createMock(EntityTypeManagerInterface::class);
    /** @var \Drupal\Core\Language\LanguageManagerInterface|\PHPUnit\Framework\MockObject\MockObject $language_manager */
    $language_manager = $this->createMock(LanguageManagerInterface::class);
    /** @var \Drupal\Core\Database\Connection|\PHPUnit\Framework\MockObject\MockObject $database */
    $database = $this->createMock(Connection::class);
    /** @var \Drupal\Core\Extension\ModuleHandlerInterface|\PHPUnit\Framework\MockObject\MockObject $module_handler */
    $module_handler = $this->createMock(ModuleHandlerInterface::class);
    /** @var \GuzzleHttp\Client|\PHPUnit\Framework\MockObject\MockObject $http_client */
    $http_client = $this->createMock(Client::class);
    /** @var \Psr\Log\LoggerInterface|\PHPUnit\Framework\MockObject\MockObject $logger */
    $logger = $this->createMock(LoggerInterface::class);

    $this->yextApiClient = new YextApiClient(
      $entity_type_manager,
      $language_manager,
      $database,
      $module_handler,
      $http_client,
      $logger
    );
  }

  /**
   * Tests helper instantiation.
   *
   * @covers ::__construct
   */
  public function testHelperInstantiation(): void {
    $this->assertInstanceOf(YextApiClient::class, $this->yextApiClient);
  }

  /**
   * Tests successful connection.
   *
   * @covers ::testConnection
   */
  public function testConnectionSuccess(): void {
    // Create a mock HTTP client that returns successful response.
    $response = new Response(200, [], '{"entities": []}');
    /** @var \GuzzleHttp\Client|\PHPUnit\Framework\MockObject\MockObject $http_client */
    $http_client = $this->createMock(Client::class);
    $http_client->method('get')->willReturn($response);

    // Create helper with mocked HTTP client.
    $yextApiClient = new YextApiClient(
      $this->createMock(EntityTypeManagerInterface::class),
      $this->createMock(LanguageManagerInterface::class),
      $this->createMock(Connection::class),
      $this->createMock(ModuleHandlerInterface::class),
      $http_client,
      $this->createMock(LoggerInterface::class)
    );

    $result = $yextApiClient->testConnection('test_account', 'test_key');
    $this->assertTrue($result);
  }

  /**
   * Tests failed connection.
   *
   * @covers ::testConnection
   */
  public function testConnectionFailure(): void {
    // Create a mock HTTP client that throws GuzzleException.
    /** @var \GuzzleHttp\Client|\PHPUnit\Framework\MockObject\MockObject $http_client */
    $http_client = $this->createMock(Client::class);
    $http_client->method('get')->willThrowException(
      new ConnectException('Connection failed', new Request('GET', 'test'))
    );

    $logger = $this->createMock(LoggerInterface::class);
    $logger->expects($this->once())->method('error');

    // Create helper with mocked HTTP client.
    $yextApiClient = new YextApiClient(
      $this->createMock(EntityTypeManagerInterface::class),
      $this->createMock(LanguageManagerInterface::class),
      $this->createMock(Connection::class),
      $this->createMock(ModuleHandlerInterface::class),
      $http_client,
      $logger
    );

    $result = $yextApiClient->testConnection('test_account', 'test_key');
    $this->assertFalse($result);
  }

}
