<?php

declare(strict_types=1);

namespace Drupal\Tests\search_api_yext\Unit;

use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\search_api_yext\YextApiClient;
use Drupal\Tests\UnitTestCase;
use GuzzleHttp\Client;
use PHPUnit\Framework\MockObject\MockObject;
use Psr\Log\LoggerInterface;
use GuzzleHttp\Exception\ClientException;
use GuzzleHttp\Exception\ServerException;
use GuzzleHttp\Psr7\Request;
use GuzzleHttp\Psr7\Response;

/**
 * Tests the YextApiClient class.
 *
 * @group search_api_yext
 * @coversDefaultClass \Drupal\search_api_yext\YextApiClient
 */
class YextApiClientTest extends UnitTestCase {

  /**
   * The YextApiClient service being tested.
   *
   * @var \Drupal\search_api_yext\YextApiClient
   */
  protected YextApiClient $yextApiClient;

  /**
   * Mock HTTP client.
   *
   * @var \GuzzleHttp\Client|\PHPUnit\Framework\MockObject\MockObject
   */
  protected Client|MockObject $httpClient;

  /**
   * Mock logger.
   *
   * @var \Psr\Log\LoggerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected LoggerInterface|MockObject $logger;

  /**
   * Mock database connection.
   *
   * @var \Drupal\Core\Database\Connection|\PHPUnit\Framework\MockObject\MockObject
   */
  protected Connection|MockObject $database;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    /** @var \GuzzleHttp\Client|\PHPUnit\Framework\MockObject\MockObject $httpClient */
    $this->httpClient = $this->createMock(Client::class);
    /** @var \Psr\Log\LoggerInterface|\PHPUnit\Framework\MockObject\MockObject $logger */
    $this->logger = $this->createMock(LoggerInterface::class);
    /** @var \Drupal\Core\Database\Connection|\PHPUnit\Framework\MockObject\MockObject $database */
    $this->database = $this->createMock(Connection::class);

    /** @var \Drupal\Core\Entity\EntityTypeManagerInterface|\PHPUnit\Framework\MockObject\MockObject $entityTypeManager */
    $entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    /** @var \Drupal\Core\Language\LanguageManagerInterface|\PHPUnit\Framework\MockObject\MockObject $languageManager */
    $languageManager = $this->createMock(LanguageManagerInterface::class);
    /** @var \Drupal\Core\Extension\ModuleHandlerInterface|\PHPUnit\Framework\MockObject\MockObject $moduleHandler */
    $moduleHandler = $this->createMock(ModuleHandlerInterface::class);

    $this->yextApiClient = new YextApiClient(
      $entityTypeManager,
      $languageManager,
      $this->database,
      $moduleHandler,
      $this->httpClient,
      $this->logger
    );
  }

  /**
   * Tests successful connection to Yext API.
   *
   * @covers ::testConnection
   */
  public function testConnectionSuccess(): void {
    $response = new Response(200, [], '{"response": {"entities": []}}');

    $this->httpClient->expects($this->once())
      ->method('get')
      ->with(
        $this->stringContains('/accounts/test_account/entities'),
        $this->callback(function ($options) {
          return $options['query']['api_key'] === 'test_key' &&
            $options['query']['limit'] === 1 &&
            isset($options['query']['v']);
        })
      )
      ->willReturn($response);

    $result = $this->yextApiClient->testConnection('test_account', 'test_key');
    $this->assertTrue($result);
  }

  /**
   * Tests failed connection to Yext API.
   *
   * @covers ::testConnection
   */
  public function testConnectionFailure(): void {
    $request = new Request('GET', 'test');
    $response = new Response(401, [], '{"error": "Unauthorized"}');
    $exception = new ClientException('Unauthorized', $request, $response);

    $this->httpClient->expects($this->once())
      ->method('get')
      ->willThrowException($exception);

    $this->logger->expects($this->once())
      ->method('error')
      ->with(
        $this->stringContains('Yext API connection test failed'),
        $this->arrayHasKey('@message')
      );

    $result = $this->yextApiClient->testConnection('test_account', 'test_key');
    $this->assertFalse($result);
  }

  /**
   * Tests successful entity indexing using connector API.
   *
   * @covers ::indexEntities
   */
  public function testIndexEntitiesSuccess(): void {
    $entities = [
      [
        'entityId' => 'test_entity_1',
        'name' => 'Test Entity',
        'description' => 'Test Description',
      ],
    ];
    $connector_name = 'test_connector';

    $response = new Response(200, [], '{"status": "success"}');
    $this->httpClient->expects($this->once())
      ->method('post')
      ->with(
        $this->stringContains('/accounts/test_account/connectors/test_connector/pushData'),
        $this->callback(function ($options) use ($entities) {
          return $options['json'] === $entities &&
          // No runMode for indexing.
            !isset($options['query']['runMode']) &&
            $options['query']['api_key'] === 'test_key' &&
            isset($options['query']['v']);
        })
      )
      ->willReturn($response);

    $result = $this->yextApiClient->indexEntities('test_account', 'test_key', $entities, $connector_name);

    $this->assertCount(1, $result);
    $this->assertEquals(200, $result[0]['status_code']);
  }

  /**
   * Tests entity indexing failure.
   *
   * @covers ::indexEntities
   */
  public function testIndexEntitiesFailure(): void {
    $entities = [['entityId' => 'test_entity_1']];
    $connector_name = 'test_connector';

    $request = new Request('POST', 'test');
    $response = new Response(500);
    $exception = new ServerException('Server Error', $request, $response);

    $this->httpClient->expects($this->once())
      ->method('post')
      ->willThrowException($exception);

    $this->logger->expects($this->once())
      ->method('error')
      ->with(
        $this->stringContains('Failed to index entity'),
        $this->arrayHasKey('@message')
      );

    $this->expectException(\Exception::class);
    $this->expectExceptionMessage('Failed to index entity to Yext');

    $this->yextApiClient->indexEntities('test_account', 'test_key', $entities, $connector_name);
  }

  /**
   * Tests successful entity deletion using connector API.
   *
   * @covers ::deleteEntities
   */
  public function testDeleteEntitiesSuccess(): void {
    $entity_ids = ['entity_1', 'entity_2'];
    $connector_name = 'test_connector';

    $response = new Response(200, [], '{"status": "deleted"}');
    $this->httpClient->expects($this->once())
      ->method('post')
      ->with(
        $this->stringContains('/accounts/test_account/connectors/test_connector/pushData'),
        $this->callback(function ($options) {
          return isset($options['query']['runMode'])
            && $options['query']['runMode'] === 'DELETION'
            && isset($options['json'])
            && count($options['json']) === 2
            && $options['json'][0]['entityId'] === 'entity_1'
            && $options['json'][1]['entityId'] === 'entity_2';
        })
      )
      ->willReturn($response);

    $this->logger->expects($this->once())
      ->method('info')
      ->with(
        $this->stringContains('Successfully requested deletion of @count entities'),
        $this->arrayHasKey('@count')
      );

    $result = $this->yextApiClient->deleteEntities('test_account', 'test_key', $connector_name, $entity_ids);

    $this->assertCount(1, $result);
    $this->assertEquals($entity_ids, $result[0]['entity_ids']);
    $this->assertEquals(200, $result[0]['status_code']);
  }

  /**
   * Tests entity deletion failure.
   *
   * @covers ::deleteEntities
   */
  public function testDeleteEntitiesFailure(): void {
    $entity_ids = ['entity_1'];
    $connector_name = 'test_connector';

    $request = new Request('POST', 'test');
    $response = new Response(500);
    $exception = new ServerException('Server Error', $request, $response);

    $this->httpClient->expects($this->once())
      ->method('post')
      ->willThrowException($exception);

    $this->logger->expects($this->once())
      ->method('error')
      ->with(
        $this->stringContains('Failed to delete entities from Yext'),
        $this->arrayHasKey('@message')
      );

    $this->expectException(\Exception::class);
    $this->expectExceptionMessage('Failed to delete entities from Yext');

    $this->yextApiClient->deleteEntities('test_account', 'test_key', $connector_name, $entity_ids);
  }

  /**
   * Tests conversion from Search API item ID to Yext entity ID.
   *
   * @covers ::convertToYextEntityId
   */
  public function testConvertToYextEntityId(): void {
    $search_api_id = 'entity:node/123:en';
    $expected = 'node-123-en';

    $result = $this->yextApiClient->convertToYextEntityId($search_api_id);
    $this->assertEquals($expected, $result);
  }

  /**
   * Tests conversion from Yext entity ID to Search API item ID.
   *
   * @covers ::convertFromYextEntityId
   */
  public function testConvertFromYextEntityId(): void {
    $yext_id = 'node-123-en';
    $expected = 'entity:node/123:en';

    $result = $this->yextApiClient->convertFromYextEntityId($yext_id);
    $this->assertEquals($expected, $result);
  }

}
