<?php

namespace Drupal\search_exclude_nid\Form;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\State\StateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * The form to collect nids to be excluded.
 */
class SearchExcludeNidForm extends FormBase {

  /**
   * The field storage config storage.
   *
   * @var \Drupal\Core\Entity\EntityTypeManager
   */
  protected $entityTypeManager;

  /**
   * The state keyvalue collection.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  private $state;

  /**
   * Creates a SearchExcludeNidForm form.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity manager service.
   * @param \Drupal\Core\State\StateInterface $state
   *   Access to State API.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, StateInterface $state) {
    $this->entityTypeManager = $entity_type_manager;
    $this->state = $state;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('state')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'search_exclude_nid_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $excluded_nids = $this->state->get('search_exclude_nid.excluded_nids');

    // Source text field.
    $form['excluded_nids'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Nodes to be excluded'),
      '#default_value' => !empty($excluded_nids) ? implode(',', $excluded_nids) : '',
      '#description' => $this->t('Comma separated list of node ids (node:nid) to be excluded, eg: 1,4,8,23 etc.'),
    ];

    $form['actions']['#type'] = 'actions';
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save settings'),
      '#button_type' => 'primary',
    ];

    $form['#theme'] = 'system_config_form';
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {

  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $excluded_nids = [];

    if (!empty($form_state->getValue('excluded_nids'))) {
      $excluded_nids_arr = explode(',', $form_state->getValue('excluded_nids'));
      foreach ($excluded_nids_arr as $excluded_nid) {
        $excluded_nid = intval($excluded_nid);
        $node = $this->entityTypeManager->getStorage('node')
          ->load($excluded_nid);

        // Check if node exists for given nid and avoid duplicates.
        if ($excluded_nid && !in_array($excluded_nid, $excluded_nids) && !empty($node)) {
          $excluded_nids[] = $excluded_nid;
        }
        else {
          $this->messenger()
            ->addWarning($this->t('nid: %nid has been removed from exclusion list as no node exists with that id or it is a duplicate.', ['%nid' => $excluded_nid]));
        }
      }
    }

    $this->state->set('search_exclude_nid.excluded_nids', $excluded_nids);
  }

}
