(function (Drupal, drupalSettings, $, once) {
  'use strict';

  /**
   * Normalizes a search term.
   */
  function normalizeTerm(term) {
    if (!term) return '';
    return term.trim().replace(/\s+/g, ' ').toLowerCase();
  }

  /**
   * Loads history from localStorage.
   */
  function loadHistory(storageKey) {
    try {
      const raw = localStorage.getItem(storageKey);
      if (!raw) return [];
      const parsed = JSON.parse(raw);
      return Array.isArray(parsed) ? parsed : [];
    } catch (e) {
      return [];
    }
  }

  /**
   * Saves history to localStorage.
   */
  function saveHistory(storageKey, list) {
    try {
      localStorage.setItem(storageKey, JSON.stringify(list));
    } catch (e) {
      // noop
    }
  }

  /**
   * Adds a term to the history.
   */
  function addTerm(storageKey, term, max) {
    const list = loadHistory(storageKey);
    
    if (list[0] === term) {
      return list; // Consecutive duplicate
    }
    
    // Remove previous duplicates to move to the beginning
    const filtered = list.filter(t => t !== term);
    filtered.unshift(term);
    
    if (filtered.length > max) {
      filtered.length = max; // truncate
    }
    
    saveHistory(storageKey, filtered);
    return filtered;
  }

  /**
   * Renders the list of terms in the block.
   */
  function renderList($container, storageKey) {
    const list = loadHistory(storageKey);
    const paramName = drupalSettings.searchTracker.paramName;
    
    if (list.length === 0) {
      $container.html('<p class="search-tracker-empty">No recent searches</p>');
      return;
    }
    
    const items = list.map(term => {
      const encoded = encodeURIComponent(term);
      const url = window.location.pathname + '?' + paramName + '=' + encoded;
      return '<li><a href="' + url + '">' + Drupal.checkPlain(term) + '</a></li>';
    });
    
    $container.html('<ul class="search-tracker-items">' + items.join('') + '</ul>');
  }

  /**
   * Main module initialization.
   */
  function init(context) {
    const settings = drupalSettings.searchTracker;
    if (!settings) return;
    
    const paramName = settings.paramName;
    const maxItems = settings.maxItems || 10;
    const storageKey = settings.storageKey || 'search_tracker_terms';

    // Detect parameter in URL
    const params = new URLSearchParams(window.location.search);
    if (params.has(paramName)) {
      const norm = normalizeTerm(params.get(paramName));
      if (norm) {
        addTerm(storageKey, norm, maxItems);
      }
    }

    // Render blocks
    const blocks = once('search-tracker', '.search-tracker-block', context);
    blocks.forEach(function(block) {
      const $block = $(block);
      const $list = $block.find('.search-tracker-list');
      
      if ($list.length === 0) return;
      
      renderList($list, storageKey);
      
      $block.on('click', '.search-tracker-clear', function (e) {
        e.preventDefault();
        localStorage.removeItem(storageKey);
        renderList($list, storageKey);
      });
    });
  }

  Drupal.behaviors.searchTracker = {
    attach: function (context) {
      init(context);
    }
  };

})(Drupal, drupalSettings, jQuery, once);
