<?php

namespace Drupal\searchify_connector\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

class SearchifyConfigForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['searchify_connector.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'searchify_connector_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('searchify_connector.settings');

    $form['base_url'] = [
      '#type' => 'url',
      '#title' => $this->t('API Base URL'),
      '#default_value' => $config->get('base_url'),
      '#required' => TRUE,
      '#description' => $this->t('Enter the base URL for the Searchify API endpoint (e.g., https://example.com/v1/stream)'),
    ];

    $form['username'] = [
      '#type' => 'textfield',
      '#title' => $this->t('API Username'),
      '#default_value' => $config->get('username'),
      '#required' => TRUE,
      '#description' => $this->t('Enter your Searchify API username. This will be stored securely in your site configuration.'),
    ];

    $form['password'] = [
      '#type' => 'password',
      '#title' => $this->t('API Password'),
      '#description' => $this->t('Enter your Searchify API password. Leave blank to keep the existing password. This will be stored securely in your site configuration.'),
    ];

    // Show a note if password is already configured
    if ($config->get('password')) {
      $form['password']['#description'] = $this->t('Enter a new password only if you wish to change it. A password is already configured.');
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('searchify_connector.settings');
    $password = $form_state->getValue('password');
    
    // If password field is empty and no password exists in config, require it
    if (empty($password) && empty($config->get('password'))) {
      $form_state->setErrorByName('password', $this->t('API password is required.'));
    }
    
    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('searchify_connector.settings');
    
    $config->set('base_url', $form_state->getValue('base_url'));
    $config->set('username', $form_state->getValue('username'));
    
    // Only update password if a new one was provided
    if (!empty($form_state->getValue('password'))) {
      $config->set('password', $form_state->getValue('password'));
    }
    
    $config->save();
    parent::submitForm($form, $form_state);
  }
}