<?php

namespace Drupal\searchify_connector\Service;

use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\RequestException;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Config\ConfigFactoryInterface;

class SearchifyApiService {

  protected $httpClient;
  protected $logger;
  protected $config;

  public function __construct(ClientInterface $http_client, LoggerChannelFactoryInterface $logger_factory, ConfigFactoryInterface $config_factory) {
    $this->httpClient = $http_client;
    $this->logger = $logger_factory->get('searchify_connector');
    $this->config = $config_factory->get('searchify_connector.settings');
  }

  public function performSearch($query) {
    $url = $this->config->get('base_url');
    $username = $this->config->get('username');
    $password = $this->config->get('password');

    // Validate that required configuration is present
    if (empty($url) || empty($username) || empty($password)) {
      $this->logger->error('Searchify API configuration is incomplete. Please configure the API settings at /admin/config/search/searchify');
      return [];
    }

    try {
      $response = $this->httpClient->request('POST', $url, [
        'auth' => [$username, $password],
        'headers' => [
          'Content-Type' => 'application/json',
          'Accept' => 'application/json',
        ],
        'json' => [
          'query' => $query,
          'stream' => false 
        ],
        'timeout' => 15,
      ]);

      $statusCode = $response->getStatusCode();
      $content = $response->getBody()->getContents();

      if ($statusCode == 200) {
        return json_decode($content, TRUE);
      }
      else {
        $this->logger->error('API returned status code: @code', ['@code' => $statusCode]);
        return [];
      }
    }
    catch (RequestException $e) {
      $this->logger->error('API Request failed: @message', ['@message' => $e->getMessage()]);
      return [];
    }
  }
}