<?php

namespace Drupal\searchify_connector\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

class SearchifyConfigForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['searchify_connector.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'searchify_connector_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('searchify_connector.settings');

    $form['base_url'] = [
      '#type' => 'url',
      '#title' => $this->t('API Base URL'),
      '#default_value' => $config->get('base_url'),
      '#required' => TRUE,
      '#description' => $this->t('Enter the base URL for the Searchify API endpoint (e.g., https://example.com/v1/stream)'),
    ];

    $form['api_key'] = [
      '#type' => 'password',
      '#title' => $this->t('API Key'),
      '#description' => $this->t('Enter your Searchify API key. This will be stored securely in your site configuration. Leave blank to keep the existing key.'),
    ];

    // Show a note if API key is already configured
    if ($config->get('api_key')) {
      $form['api_key']['#description'] = $this->t('Enter a new API key only if you wish to change it. An API key is already configured.');
    }

    $form['instance_hash'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Instance Hash'),
      '#default_value' => $config->get('instance_hash'),
      '#required' => TRUE,
      '#description' => $this->t('Enter your Searchify instance hash. This is required for API requests according to the Searchify API documentation.'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('searchify_connector.settings');
    $api_key = $form_state->getValue('api_key');
    
    // If API key field is empty and no API key exists in config, require it
    if (empty($api_key) && empty($config->get('api_key'))) {
      $form_state->setErrorByName('api_key', $this->t('API key is required.'));
    }
    
    // Validate instance_hash is provided
    $instance_hash = $form_state->getValue('instance_hash');
    if (empty($instance_hash)) {
      $form_state->setErrorByName('instance_hash', $this->t('Instance hash is required.'));
    }
    
    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('searchify_connector.settings');
    
    $config->set('base_url', $form_state->getValue('base_url'));
    $config->set('instance_hash', $form_state->getValue('instance_hash'));
    
    // Only update API key if a new one was provided
    if (!empty($form_state->getValue('api_key'))) {
      $config->set('api_key', $form_state->getValue('api_key'));
    }
    
    $config->save();
    parent::submitForm($form, $form_state);
  }
}