<?php

namespace Drupal\Tests\searchify_connector\Unit;

use PHPUnit\Framework\TestCase;
use GuzzleHttp\ClientInterface;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\StreamInterface;
use GuzzleHttp\Exception\RequestException;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\searchify_connector\Service\SearchifyApiService;

class SearchifyApiServiceTest extends TestCase {

  public function testPerformSearchReturnsDecodedJsonOn200() {
    $json = json_encode(['data' => ['one', 'two']]);

    $stream = $this->createMock(StreamInterface::class);
    $stream->method('getContents')->willReturn($json);

    $response = $this->createMock(ResponseInterface::class);
    $response->method('getStatusCode')->willReturn(200);
    $response->method('getBody')->willReturn($stream);

    $http = $this->createMock(ClientInterface::class);
    $http->expects($this->once())
         ->method('request')
         ->with('POST', $this->anything(), $this->arrayHasKey('json'))
         ->willReturn($response);

    $loggerFactory = $this->createMock(LoggerChannelFactoryInterface::class);
    $logger = $this->getMockBuilder(\Psr\Log\LoggerInterface::class)->getMock();
    $loggerFactory->method('get')->willReturn($logger);

    $config = $this->getMockBuilder(\Drupal\Core\Config\ImmutableConfig::class)
                   ->disableOriginalConstructor()
                   ->getMock();
    $config->method('get')->willReturnMap([
      ['base_url', 'https://example.local/v1/stream'],
      ['username', 'user'],
      ['password', 'pass'],
    ]);

    $configFactory = $this->createMock(ConfigFactoryInterface::class);
    $configFactory->method('get')->with('searchify_connector.settings')->willReturn($config);

    $service = new SearchifyApiService($http, $loggerFactory, $configFactory);
    $result = $service->performSearch('test');

    $this->assertIsArray($result);
    $this->assertArrayHasKey('data', $result);
    $this->assertEquals(['one', 'two'], $result['data']);
  }

  public function testPerformSearchHandlesRequestExceptionAndReturnsEmpty() {
    $http = $this->createMock(ClientInterface::class);
    $http->method('request')->willThrowException(new RequestException('error', $this->createMock(\Psr\Http\Message\RequestInterface::class)));

    $loggerFactory = $this->createMock(LoggerChannelFactoryInterface::class);
    $logger = $this->getMockBuilder(\Psr\Log\LoggerInterface::class)->getMock();
    $loggerFactory->method('get')->willReturn($logger);

    $config = $this->getMockBuilder(\Drupal\Core\Config\ImmutableConfig::class)
                   ->disableOriginalConstructor()
                   ->getMock();
    $config->method('get')->willReturnMap([
      ['base_url', NULL],
      ['username', NULL],
      ['password', NULL],
    ]);

    $configFactory = $this->createMock(ConfigFactoryInterface::class);
    $configFactory->method('get')->with('searchify_connector.settings')->willReturn($config);

    $service = new SearchifyApiService($http, $loggerFactory, $configFactory);
    $result = $service->performSearch('test');

    $this->assertIsArray($result);
    $this->assertEmpty($result);
  }
}
