#!/bin/bash

# Quick Test Script for Searchify Connector Module
# This script helps you quickly test your module locally

echo "=========================================="
echo "Searchify Connector - Quick Test Setup"
echo "=========================================="
echo ""

# Check if we're in a Drupal site
if [ ! -f "composer.json" ] && [ ! -f "web/core/version.php" ]; then
    echo "⚠️  Warning: This doesn't appear to be a Drupal site directory."
    echo "   Please run this script from your Drupal site root."
    echo ""
    echo "   Or use one of these methods:"
    echo "   1. Add path repository to your site's composer.json"
    echo "   2. Clone the module to web/modules/custom/"
    echo ""
    read -p "Continue anyway? (y/n) " -n 1 -r
    echo
    if [[ ! $REPLY =~ ^[Yy]$ ]]; then
        exit 1
    fi
fi

MODULE_PATH="/Users/nikhiluppal/Downloads/searchify_connector"

echo "📦 Module location: $MODULE_PATH"
echo ""

# Check if module directory exists
if [ ! -d "$MODULE_PATH" ]; then
    echo "❌ Error: Module directory not found at $MODULE_PATH"
    exit 1
fi

echo "Choose installation method:"
echo "1) Composer path repository (recommended for development)"
echo "2) Direct symlink to modules/custom"
echo "3) Show manual instructions"
echo ""
read -p "Enter choice (1-3): " choice

case $choice in
    1)
        echo ""
        echo "📝 Adding path repository to composer.json..."
        
        # Check if composer.json exists
        if [ ! -f "composer.json" ]; then
            echo "❌ composer.json not found. Are you in a Drupal site directory?"
            exit 1
        fi
        
        # Check if repository already exists
        if grep -q "searchify_connector" composer.json; then
            echo "⚠️  Path repository may already exist in composer.json"
        else
            # Create backup
            cp composer.json composer.json.backup
            
            # Add repository (simple approach - you may need to manually edit)
            echo ""
            echo "📝 Please manually add this to your composer.json 'repositories' section:"
            echo ""
            echo "   {"
            echo "     \"type\": \"path\","
            echo "     \"url\": \"$MODULE_PATH\""
            echo "   },"
            echo ""
            echo "Then run: composer require drupal/searchifyai:@dev"
        fi
        ;;
    2)
        echo ""
        echo "🔗 Creating symlink..."
        
        MODULES_DIR="web/modules/custom"
        if [ ! -d "$MODULES_DIR" ]; then
            echo "Creating $MODULES_DIR..."
            mkdir -p "$MODULES_DIR"
        fi
        
        SYMLINK_PATH="$MODULES_DIR/searchify_connector"
        
        if [ -L "$SYMLINK_PATH" ] || [ -d "$SYMLINK_PATH" ]; then
            echo "⚠️  $SYMLINK_PATH already exists"
            read -p "Remove and recreate? (y/n) " -n 1 -r
            echo
            if [[ $REPLY =~ ^[Yy]$ ]]; then
                rm -rf "$SYMLINK_PATH"
            else
                exit 0
            fi
        fi
        
        ln -s "$MODULE_PATH" "$SYMLINK_PATH"
        echo "✅ Symlink created: $SYMLINK_PATH -> $MODULE_PATH"
        echo ""
        echo "Now enable the module:"
        echo "  drush en searchify_connector -y"
        echo "  drush cr"
        ;;
    3)
        echo ""
        echo "📖 Manual Instructions:"
        echo ""
        echo "Method 1 - Composer Path Repository:"
        echo "  1. Add to your site's composer.json 'repositories' section:"
        echo "     {"
        echo "       \"type\": \"path\","
        echo "       \"url\": \"$MODULE_PATH\""
        echo "     }"
        echo "  2. Run: composer require drupal/searchifyai:@dev"
        echo "  3. Run: drush en searchify_connector -y && drush cr"
        echo ""
        echo "Method 2 - Direct Clone:"
        echo "  1. cd web/modules/custom"
        echo "  2. git clone https://git.drupalcode.org/project/searchifyai.git searchify_connector"
        echo "  3. drush en searchify_connector -y && drush cr"
        echo ""
        echo "Method 3 - Composer from Drupal.org (after release):"
        echo "  1. composer require drupal/searchifyai"
        echo "  2. drush en searchify_connector -y && drush cr"
        echo ""
        echo "Test URLs:"
        echo "  - Search: http://your-site.local/searchify"
        echo "  - Config: http://your-site.local/admin/config/search/searchify"
        ;;
    *)
        echo "Invalid choice"
        exit 1
        ;;
esac

echo ""
echo "=========================================="
echo "✅ Setup complete!"
echo "=========================================="
echo ""
echo "Next steps:"
echo "1. Enable module: drush en searchify_connector -y"
echo "2. Clear cache: drush cr"
echo "3. Configure: /admin/config/search/searchify"
echo "4. Test search: /searchify"
echo ""


