/**
 * @file
 * JavaScript for Searchify Connector search results.
 */

(function (Drupal) {
  'use strict';

  /**
   * Process citation links in the response text.
   */
  Drupal.behaviors.searchifyCitations = {
    attach: function (context, settings) {
      const responseText = document.querySelector('.searchify-text');
      if (!responseText) {
        return;
      }

      // Find all citation markers like [0], [1], [2], etc.
      // According to Searchify API docs: citations appear as [0], [1], [2], etc.
      const citationRegex = /\[(\d+)\]/g;
      let html = responseText.innerHTML;

      // First pass: Replace citation markers with clickable links
      // We'll validate targets after HTML is inserted
      html = html.replace(citationRegex, function(match, citationId) {
        // Ensure citationId is treated as string to match template IDs
        const citationIdStr = String(citationId);
        const targetId = 'citation-' + citationIdStr;
        
        return '<a href="#' + targetId + '" class="citation-link" title="Jump to source ' + citationIdStr + '" data-citation-id="' + citationIdStr + '">[' + citationIdStr + ']</a>';
      });

      responseText.innerHTML = html;

      // Second pass: Validate links and add click handlers
      // Now that HTML is inserted, we can check if targets exist
      const citationLinks = responseText.querySelectorAll('a.citation-link');
      citationLinks.forEach(function(link) {
        const citationId = link.getAttribute('data-citation-id');
        const targetId = 'citation-' + citationId;
        const targetElement = document.getElementById(targetId);
        
        if (!targetElement) {
          // If target doesn't exist, mark as invalid
          link.classList.add('citation-link-invalid');
          link.style.cursor = 'not-allowed';
          link.title = 'Source not found';
        }
        
        // Add click handler for smooth scrolling
        link.addEventListener('click', function(e) {
          const clickedCitationId = this.getAttribute('data-citation-id');
          const clickedTargetId = 'citation-' + clickedCitationId;
          const clickedTargetElement = document.getElementById(clickedTargetId);
          
          if (clickedTargetElement) {
            // Smooth scroll to the citation
            e.preventDefault();
            clickedTargetElement.scrollIntoView({ behavior: 'smooth', block: 'start' });
            
            // Update URL hash without triggering scroll
            if (history.pushState) {
              history.pushState(null, null, '#' + clickedTargetId);
            }
            
            // Highlight the target briefly
            clickedTargetElement.classList.add('citation-highlight');
            setTimeout(function() {
              clickedTargetElement.classList.remove('citation-highlight');
            }, 2000);
          } else {
            // If target doesn't exist, prevent navigation
            e.preventDefault();
          }
        });
      });
    }
  };

})(Drupal);


