<?php

namespace Drupal\searchify_connector\Service;

use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\RequestException;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Config\ConfigFactoryInterface;

class SearchifyApiService {

  protected $httpClient;
  protected $logger;
  protected $config;

  public function __construct(ClientInterface $http_client, LoggerChannelFactoryInterface $logger_factory, ConfigFactoryInterface $config_factory) {
    $this->httpClient = $http_client;
    $this->logger = $logger_factory->get('searchify_connector');
    $this->config = $config_factory->get('searchify_connector.settings');
  }

  public function performSearch($query) {
    $url = $this->config->get('base_url');
    $api_key = $this->config->get('api_key');
    $instance_hash = $this->config->get('instance_hash');

    // Validate that required configuration is present
    // According to https://github.com/searchify-it/searchify-api-docs:
    // - API key (Bearer token) is required for authentication
    // - instance_hash is required in the request body
    if (empty($url) || empty($api_key) || empty($instance_hash)) {
      $this->logger->error('Searchify API configuration is incomplete. Please configure the API settings at /admin/config/search/searchify');
      return [];
    }

    try {
      // According to Searchify API docs: https://github.com/searchify-it/searchify-api-docs
      // Request format: { "instance_hash": "...", "query": "..." }
      // Authentication: Authorization: Bearer {api_key}
      // Response: Server-Sent Events (SSE) with JSON data
      $response = $this->httpClient->request('POST', $url, [
        'headers' => [
          'Authorization' => 'Bearer ' . $api_key,
          'Content-Type' => 'application/json',
          'Accept' => 'text/event-stream',
        ],
        'json' => [
          'instance_hash' => $instance_hash,
          'query' => $query,
        ],
        'stream' => true,
        'timeout' => 30,
      ]);

      $statusCode = $response->getStatusCode();

      if ($statusCode == 200) {
        // Handle Server-Sent Events (SSE) streaming response
        return $this->parseSSEResponse($response);
      }
      else {
        $this->logger->error('API returned status code: @code', ['@code' => $statusCode]);
        return [];
      }
    }
    catch (RequestException $e) {
      $this->logger->error('API Request failed: @message', ['@message' => $e->getMessage()]);
      return [];
    }
  }

  /**
   * Parse Server-Sent Events (SSE) response from Searchify API.
   *
   * @param \Psr\Http\Message\ResponseInterface $response
   *   The HTTP response.
   *
   * @return array
   *   Parsed response data with text and search results.
   */
  protected function parseSSEResponse($response) {
    $body = $response->getBody();
    $fullText = '';
    $searchResults = [];
    $isDone = false;
    $buffer = '';

    // Read the stream chunk by chunk
    while (!$body->eof()) {
      $chunk = $body->read(8192); // Read 8KB at a time
      if ($chunk === false || $chunk === '') {
        break;
      }
      
      $buffer .= $chunk;
      
      // Process complete lines
      while (($pos = strpos($buffer, "\n")) !== false) {
        $line = substr($buffer, 0, $pos);
        $buffer = substr($buffer, $pos + 1);
        
        // Skip empty lines and non-data lines
        $line = trim($line);
        if (empty($line) || !str_starts_with($line, 'data: ')) {
          continue;
        }

        // Extract JSON data (remove 'data: ' prefix)
        $jsonData = substr($line, 6);
        $data = json_decode($jsonData, TRUE);

        if ($data && json_last_error() === JSON_ERROR_NONE) {
          // updatedText contains the complete response text so far
          if (isset($data['updatedText'])) {
            $fullText = $data['updatedText'];
          }

          // Collect search results
          // According to API docs: id can be "0", "1", "2", etc. (strings)
          // Store with original ID as key to preserve citation mapping
          if (isset($data['search_results']) && is_array($data['search_results'])) {
            foreach ($data['search_results'] as $result) {
              // Use result ID as key to avoid duplicates
              // Ensure ID is stored as string to match citation format [0], [1], etc.
              if (isset($result['id'])) {
                $id = (string) $result['id']; // Normalize to string
                $result['id'] = $id; // Ensure consistent string format
                
                // Truncate snippet parts to limit length (like demo site)
                if (isset($result['snippet'])) {
                  $result['snippet'] = $this->truncateSnippet($result['snippet']);
                }
                
                $searchResults[$id] = $result;
              }
            }
          }

          // Check if streaming is complete
          if (isset($data['isDone']) && $data['isDone']) {
            $isDone = true;
            break 2; // Break out of both loops
          }
        }
      }
    }

    // Extract citation IDs in the order they appear in the text
    // This ensures proper sequential numbering [0], [1], [2], etc.
    $citedIdsInOrder = [];
    if (!empty($fullText)) {
      // Match citation patterns like [0], [1], [2], [10], etc. in order of appearance
      if (preg_match_all('/\[(\d+)\]/', $fullText, $matches, PREG_SET_ORDER | PREG_OFFSET_CAPTURE)) {
        // Sort by position in text to maintain order
        usort($matches, function($a, $b) {
          return $a[0][1] <=> $b[0][1];
        });
        
        // Extract unique IDs in order of first appearance
        $seen = [];
        foreach ($matches as $match) {
          $citedId = $match[1][0];
          if (!in_array($citedId, $seen)) {
            $citedIdsInOrder[] = $citedId;
            $seen[] = $citedId;
          }
        }
      }
    }

    // Create mapping from original citation IDs to sequential IDs [0], [1], [2], etc.
    $citationMapping = [];
    $sequentialResults = [];
    foreach ($citedIdsInOrder as $newIndex => $originalId) {
      $originalIdStr = (string) $originalId;
      if (isset($searchResults[$originalIdStr])) {
        $citationMapping[$originalIdStr] = (string) $newIndex;
        
        // Create result with new sequential ID
        $result = $searchResults[$originalIdStr];
        $result['id'] = (string) $newIndex;
        $sequentialResults[] = $result;
      }
    }

    // If no citations found, return all results (fallback) with sequential IDs
    if (empty($citationMapping)) {
      $sequentialResults = [];
      $index = 0;
      foreach ($searchResults as $result) {
        $result['id'] = (string) $index;
        $sequentialResults[] = $result;
        $index++;
      }
    } else {
      // Replace citation markers in text with sequential numbers
      // Use preg_replace_callback to safely replace all citations at once
      $fullText = preg_replace_callback('/\[(\d+)\]/', function($matches) use ($citationMapping) {
        $originalId = $matches[1];
        if (isset($citationMapping[$originalId])) {
          return '[' . $citationMapping[$originalId] . ']';
        }
        return $matches[0]; // Return original if not in mapping
      }, $fullText);
    }

    // Return the final response with HTML formatting preserved
    return [
      'text' => $fullText, // HTML formatting is preserved
      'search_results' => $sequentialResults,
      'isDone' => $isDone,
    ];
  }

  /**
   * Truncate snippet parts to limit length (matching demo site behavior).
   *
   * @param array $snippet
   *   Snippet array with 'pre', 'text', 'post' keys.
   * @param int $maxLength
   *   Maximum length for each snippet part.
   *
   * @return array
   *   Truncated snippet array.
   */
  protected function truncateSnippet($snippet, $maxLength = 150) {
    $truncated = [];
    
    if (isset($snippet['pre'])) {
      $truncated['pre'] = $this->truncateText($snippet['pre'], $maxLength);
    }
    
    if (isset($snippet['text'])) {
      $truncated['text'] = $this->truncateText($snippet['text'], $maxLength);
    }
    
    if (isset($snippet['post'])) {
      $truncated['post'] = $this->truncateText($snippet['post'], $maxLength);
    }
    
    return $truncated;
  }

  /**
   * Truncate text to a maximum length with ellipsis.
   *
   * @param string $text
   *   Text to truncate.
   * @param int $maxLength
   *   Maximum length (including ellipsis).
   *
   * @return string
   *   Truncated text.
   */
  protected function truncateText($text, $maxLength = 150) {
    if (mb_strlen($text) <= $maxLength) {
      return $text;
    }
    
    // Truncate and add ellipsis, ensuring total length doesn't exceed maxLength
    $ellipsis = '...';
    $truncateLength = $maxLength - mb_strlen($ellipsis);
    return mb_substr($text, 0, $truncateLength) . $ellipsis;
  }
}