<?php

namespace Drupal\Tests\searchstax\Functional;

use Drupal\language\Entity\ConfigurableLanguage;
use Drupal\search_api\Entity\Server;
use Drupal\searchstax_test_mock_http\MockHttpTestTrait;
use Drupal\Tests\BrowserTestBase;

// cspell:ignore défaut deuxième drittes erstes troisième zweites

/**
 * Tests selecting the relevance model for SearchStax views.
 *
 * @group searchstax
 */
class ViewsSelectRelevanceModelTest extends BrowserTestBase {

  use MockHttpTestTrait;
  use TestAssertionsTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'search_api',
    'search_api_solr',
    'searchstax',
    'searchstax_test',
    'searchstax_test_mock_http',
    'dblog',
    'language',
    'views',
    'views_ui',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Prepare the mock HTTP client.
    $this->setDataDirectory(__DIR__ . '/../../data/views-relevance-model');

    // We want to get actual HTTP requests (to our mock client) from the Solr
    // connector, so do not use the test connector.
    $server = Server::load('searchstax_server');
    $backend_config = $server->getBackendConfig();
    $backend_config['connector'] = 'searchstax';
    $backend_config['connector_config']['update_endpoint'] = 'https://searchcloud-2-us-east-1.searchstax.com/12345/searchstax-test/update';
    $backend_config['connector_config']['update_token'] = '0123456789abcdef0123456789abcdef01234567';
    $backend_config['connector_config']['context'] = '12345';
    $server->setBackendConfig($backend_config);
    $server->save();

    // Add extra languages.
    ConfigurableLanguage::createFromLangcode('de')->save();
    ConfigurableLanguage::createFromLangcode('fr')->save();

    // Set a predictable site hash.
    \Drupal::state()->set('search_api_solr.site_hash', '123456');

    // Enable routing of searches to the /emselect handler.
    \Drupal::configFactory()->getEditable('searchstax.settings')
      ->set('searches_via_searchstudio', TRUE)
      ->set('configure_via_searchstudio', FALSE)
      ->save();

    // Disable live preview.
    \Drupal::configFactory()->getEditable('views.settings')
      ->set('ui.always_live_preview', FALSE)
      ->save();
  }

  /**
   * Tests selecting the relevance model for SearchStax views.
   *
   * @see views.view.searchstax_test_view.yml
   */
  public function testRelevanceModelOption(): void {
    $assert = $this->assertSession();
    $this->drupalLogin($this->drupalCreateUser([], NULL, TRUE));
    // Not sure why this is needed here, but in GitLab CI the search view page
    // is sometimes not found without first clearing the cache.
    drupal_flush_all_caches();
    $ignored_requests = ['core-info', 'solr-ping'];

    $this->drupalGet('test-search-view');
    $this->assertHttpRequests(['empty-search-no-model'], $ignored_requests);
    $this->assertNoErrorsOrWarnings();

    $this->drupalGet('test-search-view', ['query' => ['search_api_fulltext' => 'test']]);
    $this->assertHttpRequests(['keyword-search-no-model'], $ignored_requests);
    $this->assertNoErrorsOrWarnings();

    $view_edit_path = 'admin/structure/views/view/searchstax_test_view/edit/default';
    $this->drupalGet($view_edit_path);
    $this->click('#views-default-query');
    $this->assertNoWarningsLogged();
    $assert->pageTextNotContains('The website encountered an unexpected error.');
    $assert->pageTextNotContains('Error message');

    $assert->pageTextContains('SearchStax settings');
    $assert->pageTextContains('You need to log in to the SearchStax app in order to change these settings.');
    $assert->pageTextContains('SearchStax login');
    $this->submitForm([
      'query[options][searchstax][login][password]' => 'password123',
      'query[options][searchstax][login][username]' => 'user@example.com',
      'query[options][searchstax][login][tfa_token]' => '123456',
    ], 'Continue');
    $this->assertHttpRequests([
      'get-account-1',
      'get-models-de',
      'get-models-en',
      'get-models-fr',
      'list-accounts',
      'obtain-auth-token',
    ]);
    $this->assertNoErrorsOrWarnings();

    // The functionality to stay in the dialog after a successful login doesn't
    // work without Javascript. No matter, we just click on the "Query settings"
    // link again.
    if (strpos($this->getSession()->getCurrentUrl(), $view_edit_path) !== FALSE) {
      $this->click('#views-default-query');
    }

    $assert->pageTextContains('Relevance model for English');
    $assert->pageTextContains('default (default)');
    $assert->pageTextContains('experiment1');
    $assert->pageTextNotContains('experiment2');
    $assert->pageTextContains('experiment3');
    $assert->elementExists('css', 'input[type="radio"][value="1001"]');
    $assert->elementExists('css', 'input[type="radio"][value="1002"]');
    $assert->elementNotExists('css', 'input[type="radio"][value="1003"]');
    $assert->elementExists('css', 'input[type="radio"][value="1004"]');
    $assert->pageTextContains('Relevance model for German');
    $assert->pageTextContains('Standard (default)');
    $assert->pageTextContains('Erstes Experiment');
    $assert->pageTextNotContains('Zweites Experiment');
    $assert->pageTextContains('Drittes Experiment');
    $assert->elementExists('css', 'input[type="radio"][value="2001"]');
    $assert->elementExists('css', 'input[type="radio"][value="2002"]');
    $assert->elementNotExists('css', 'input[type="radio"][value="2003"]');
    $assert->elementExists('css', 'input[type="radio"][value="2004"]');
    $assert->pageTextContains('Relevance model for French');
    $assert->pageTextContains('défaut (default)');
    $assert->pageTextContains('Premier test');
    $assert->pageTextNotContains('Deuxième test');
    $assert->pageTextContains('Troisième test');
    $assert->elementExists('css', 'input[type="radio"][value="3001"]');
    $assert->elementExists('css', 'input[type="radio"][value="3002"]');
    $assert->elementNotExists('css', 'input[type="radio"][value="3003"]');
    $assert->elementExists('css', 'input[type="radio"][value="3004"]');
    $this->submitForm([
      'query[options][searchstax][relevance_models][de]' => '2001',
      'query[options][searchstax][relevance_models][en]' => '1002',
      'query[options][searchstax][relevance_models][fr]' => '',
    ], 'Apply');
    $this->assertNoErrorsOrWarnings();

    $this->submitForm([], 'Save');
    $this->assertNoErrorsOrWarnings();

    \Drupal::keyValue('searchstax_test_mock_http')->set('requests', []);
    $this->drupalGet('test-search-view');
    $this->assertHttpRequests(['empty-search-model-1002'], $ignored_requests);
    $this->assertNoErrorsOrWarnings();

    $this->drupalGet('test-search-view', [
      'query' => [
        'search_api_language' => 'en',
      ],
    ]);
    $this->assertHttpRequests(['empty-search-model-1002'], $ignored_requests);
    $this->assertNoErrorsOrWarnings();

    $this->drupalGet('test-search-view', [
      'query' => [
        'search_api_fulltext' => 'test',
      ],
    ]);
    $this->assertHttpRequests(['keyword-search-model-1002'], $ignored_requests);
    $this->assertNoErrorsOrWarnings();

    $this->drupalGet('test-search-view', [
      'query' => [
        'search_api_language' => 'de',
      ],
    ]);
    $this->assertHttpRequests(['empty-search-de'], $ignored_requests);
    $this->assertNoErrorsOrWarnings();

    $this->drupalGet('test-search-view', [
      'query' => [
        'search_api_language' => 'fr',
      ],
    ]);
    $this->assertHttpRequests(['empty-search-fr'], $ignored_requests);
    $this->assertNoErrorsOrWarnings();
  }

  /**
   * Asserts that no errors or warnings were displayed or logged.
   */
  protected function assertNoErrorsOrWarnings(): void {
    $assert = $this->assertSession();
    $this->assertNoWarningsLogged();
    $assert->pageTextNotContains('The website encountered an unexpected error.');
    $assert->pageTextNotContains('Error message');
    $assert->pageTextNotContains('Warning message');
  }

}
