<?php

namespace Drupal\searchstax\Plugin\SolrConnector;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\search_api\Backend\BackendInterface;
use Drupal\search_api\ServerInterface;
use Drupal\search_api_solr\Controller\SolrConfigSetController;
use Drupal\search_api_solr\Plugin\SolrConnector\StandardSolrConnector;
use Drupal\search_api_solr\SearchApiSolrException;
use Solarium\Core\Client\Endpoint;
use Solarium\Core\Client\Response;
use Solarium\Core\Query\Result\ResultInterface;
use Solarium\Exception\HttpException;
use Solarium\QueryType\Update\Query\Command\Add;
use Solarium\QueryType\Update\Query\Query as UpdateQuery;
use Symfony\Component\DependencyInjection\ContainerInterface;

// cspell:ignore configset solrcore ulog

/**
 * Provides a plugin for connecting to a SearchStax Solr server with token auth.
 *
 * @SolrConnector(
 *   id = "searchstax",
 *   label = @Translation("SearchStax Cloud with Token Auth"),
 *   description = @Translation("Index items protected by token authentication for SearchStax."),
 * )
 */
class SearchStaxConnector extends StandardSolrConnector {

  /**
   * The minimum Solr version SearchStax might use.
   */
  public const SEARCHSTAX_MINIMUM_SOLR_VERSION = '8.11.1';

  /**
   * The maximum request size allowed by SearchStax servers, in bytes.
   */
  public const SEARCHSTAX_MAX_REQUEST_SIZE = 10485760;

  /**
   * Cached version strings, keyed by server ID.
   *
   * @var array<string, string>
   */
  protected static array $versionStrings = [];

  /**
   * Cached Solr config files, keyed by server ID.
   *
   * @var array<string, array<string, string>>
   */
  protected static array $cachedFiles = [];

  /**
   * The server to which this connector plugin is linked.
   */
  protected ServerInterface $server;

  /**
   * The Solr config set controller.
   */
  protected SolrConfigSetController $solrConfigSetController;

  /**
   * The entity type manager.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Constructs a new class instance.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\search_api_solr\Controller\SolrConfigSetController $solr_config_set_controller
   *   The Solr config set controller.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    array $plugin_definition,
    SolrConfigSetController $solr_config_set_controller,
    EntityTypeManagerInterface $entity_type_manager
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);

    $this->solrConfigSetController = $solr_config_set_controller;
    $this->entityTypeManager = $entity_type_manager;

    // Try to retrieve the server entity from the call stack.
    $this->setServerFromBacktrace();
  }

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition
  ): self {
    if ($container->has('search_api_solr.configset_controller')) {
      $solr_config_set_controller = $container->get('search_api_solr.configset_controller');
    }
    else {
      $solr_config_set_controller = new SolrConfigSetController($container->get('extension.list.module'));
    }
    $plugin = new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $solr_config_set_controller,
      $container->get('entity_type.manager'),
    );

    /** @var \Psr\Log\LoggerInterface $logger */
    $logger = $container->get('logger.channel.searchstax');
    $plugin->setLogger($logger);

    return $plugin;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'scheme' => 'https',
      'host' => '',
      'port' => 443,
      'context' => '',
      'core' => '',
      self::INDEX_TIMEOUT => 15,
      'update_endpoint' => '',
      'update_token' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form['update_endpoint'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Endpoint'),
      '#description' => $this->t('Just copy &amp; paste the “Update endpoint” value of the SearchStax app / index as shown in your SearchStax account.'),
      '#default_value' => $this->configuration['update_endpoint'] ?? '',
      '#required' => TRUE,
    ];

    $form['update_token'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Read &amp; write token key'),
      '#description' => $this->t('Just copy &amp; paste the “Read & Write token key” value of the SearchStax app / index as shown in your SearchStax account.'),
      '#default_value' => $this->configuration['update_token'] ?? '',
      '#required' => TRUE,
    ];

    $form += parent::buildConfigurationForm($form, $form_state);

    $form['scheme'] = [
      '#type' => 'value',
      '#value' => 'https',
    ];

    $form['host'] = [
      '#type' => 'value',
      '#value' => '',
    ];

    $form['port'] = [
      '#type' => 'value',
      '#value' => '443',
    ];

    $form['path'] = [
      '#type' => 'value',
      '#value' => '/',
    ];

    $form['core'] = [
      '#type' => 'value',
      '#value' => '',
    ];

    $form['context'] = [
      '#type' => 'value',
      '#value' => '',
    ];

    $form['advanced']['jmx'] = [
      '#type' => 'value',
      '#value' => FALSE,
    ];

    $form['advanced']['solr_install_dir'] = [
      '#type' => 'value',
      '#value' => '',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $values = $form_state->getValues();
    $values['update_endpoint'] = trim($values['update_endpoint']);
    $form_state->setValue('update_endpoint', $values['update_endpoint']);
    $values['update_token'] = trim($values['update_token']);
    $form_state->setValue('update_token', $values['update_token']);

    if (preg_match('@https://([^/:]+)/([^/]+)/([^/]+)/(update|select)$@', $values['update_endpoint'], $matches)) {
      $form_state->setValue('host', $matches[1]);
      $form_state->setValue('context', $matches[2]);
      $form_state->setValue('core', $matches[3]);
    }
    else {
      $form_state->setErrorByName('update_endpoint', $this->t('Invalid endpoint format'));
    }

    if (empty($values['update_token'])) {
      $form_state->setErrorByName('update_token', $this->t('Invalid token format'));
    }

    // Check that the necessary method for setting the authorization token
    // exists.
    if (!method_exists(Endpoint::class, 'setAuthorizationToken')) {
      $form_state->setErrorByName('update_token', $this->t('The version of the Solarium library installed on your site does not support token authentication. Upgrade to a version newer than @version to use this Solr connector.', ['@version' => '6.2.7']));
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function connect(): void {
    if (!$this->solr) {
      parent::connect();
      $this->solr->getEndpoint('search_api_solr')
        ->setAuthorizationToken('Token', $this->configuration['update_token']);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function pingServer() {
    return $this->pingCore();
  }

  /**
   * {@inheritdoc}
   */
  public function getSolrVersion($force_auto_detect = FALSE) {
    if (!$force_auto_detect && !empty($this->configuration['solr_version'])) {
      return parent::getSolrVersion($force_auto_detect);
    }

    try {
      // The response from the "/[CORE]/admin/system" has a different structure
      // for SearchStax servers than Solr normally uses. We need to adapt
      // accordingly.
      $info = $this->getCoreInfo();
      if (!empty($info['solr_version'])) {
        return $info['solr_version'];
      }
    }
    catch (SearchApiSolrException $e) {
      // At this point, the parent method would call getServerInfo() as a
      // fallback. However, we know that that endpoint is blocked by SearchStax,
      // so no need to even try.
    }
    // As the fallback, return the minimum Solr version that might be used by
    // SearchStax. Otherwise, the Solr backend will fall back to compatibility
    // with Solr 6, which would lead to the use of deprecated field types in the
    // config-set.
    return self::SEARCHSTAX_MINIMUM_SOLR_VERSION;
  }

  /**
   * {@inheritdoc}
   */
  public function getServerInfo($reset = FALSE): array {
    // The "/admin/info/system" is blocked by SearchStax, which is why we
    // override getSolrVersion() to not even try and use it. However, if this
    // method is still invoked, still attempt to return something useful.
    return ['lucene' => ['solr-spec-version' => $this->getSolrVersion()]];
  }

  /**
   * {@inheritdoc}
   */
  public function getSchemaVersionString($reset = FALSE): string {
    $server_id = $this->getServer()->id();
    if (!isset(static::$versionStrings[$server_id])) {
      static::$versionStrings[$server_id] = 'drupal-0.0.0-solr-8.x';

      $this->connect();
      $query = $this->solr->createApi([
        'handler' => $this->configuration['core'] . '/schema',
      ]);

      if ($response = $this->execute($query)->getResponse()) {
        $body = json_decode($response->getBody(), TRUE);
        if (isset($body['schema']['name'])) {
          static::$versionStrings[$server_id] = $body['schema']['name'];
        }
      }
    }

    return static::$versionStrings[$server_id];
  }

  /**
   * {@inheritdoc}
   */
  public function coreRestGet($path, ?Endpoint $endpoint = NULL): array {
    return [
      'fieldTypes' => [
        [
          'name' => 'Information about fieldTypes is not provided by SearchStax',
        ],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function reloadCore(): bool {
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function getLuke(): array {
    return [
      'fields' => [],
      'index' => ['numDocs' => -1],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getStatsSummary(): array {
    return [
      '@pending_docs' => 0,
      '@index_size' => 0,
      '@schema_version' => $this->getSchemaVersionString(),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function update(UpdateQuery $query, ?Endpoint $endpoint = NULL): ResultInterface {
    // Guard against HTTP errors caused by timeouts or oversized requests.
    try {
      return parent::update($query, $endpoint);
    }
    catch (SearchApiSolrException $e) {
      $previous = $e->getPrevious();
      if (
        $previous instanceof HttpException
        && in_array($previous->getCode(), [CURLE_OPERATION_TIMEDOUT, 413])
      ) {
        $timeout_docs = NULL;
        if ($previous->getCode() === CURLE_OPERATION_TIMEDOUT) {
          $timeout_docs = 0;
          foreach ($query->getCommands() as $command) {
            if ($command instanceof Add) {
              $timeout_docs += count($command->getDocuments());
            }
          }
        }
        return $this->updateFallback($query, $endpoint, $timeout_docs);
      }
      throw $e;
    }
  }

  /**
   * Executes an update query, splitting the request according to size limit.
   *
   * @param \Solarium\QueryType\Update\Query\Query $query
   *   The Solarium update query object.
   * @param \Solarium\Core\Client\Endpoint|null $endpoint
   *   (optional) The Solarium endpoint object.
   * @param int|null $timeout_docs
   *   (optional) If the previous update request failed due to a timeout, the
   *   number of documents for which it failed; NULL otherwise.
   * @param int $retry_level
   *   (optional) Internal use only. Counts the number of nested calls to
   *   prevent an infinite recursion.
   *
   * @return \Solarium\Core\Query\Result\ResultInterface
   *   The Solarium result object.
   *
   * @throws \Drupal\search_api_solr\SearchApiSolrException
   *   Thrown in case of any HTTP errors.
   */
  protected function updateFallback(
    UpdateQuery $query,
    ?Endpoint $endpoint = NULL,
    ?int $timeout_docs = NULL,
    int $retry_level = 0
  ): ResultInterface {
    $docs = [];
    $other_commands = [];
    $last_result = NULL;
    foreach ($query->getCommands() as $command) {
      if ($command instanceof Add) {
        $docs = array_merge($docs, $command->getDocuments());
      }
      else {
        $other_commands[] = $command;
      }
    }

    if ($other_commands) {
      $new_update_query = $this->getUpdateQuery();
      foreach ($other_commands as $command) {
        $new_update_query->add(NULL, $command);
      }
      $last_result = parent::update($new_update_query, $endpoint);
    }

    if ($timeout_docs === NULL) {
      $log_success = function (int $count): void {
        $this->getLogger()->info("Indexed a batch of @count documents in fallback after the initial indexing request exceeded the SearchStax server's maximum request size of @limit bytes.", [
          '@count' => $count,
          '@limit' => static::SEARCHSTAX_MAX_REQUEST_SIZE,
        ]);
      };
    }
    else {
      $log_success = function (int $count): void {
        $this->getLogger()->info("Indexed a batch of @count documents in fallback after the initial indexing request hit the index timeout of @timeout seconds.", [
          '@count' => $count,
          '@timeout' => $this->configuration[self::INDEX_TIMEOUT],
        ]);
      };
    }

    $batch = [];
    // The overhead is 1 byte/char as the basis (final "}") plus 15 bytes per
    // document (","/"{" and '"add":{"doc":' before the doc and "}" afterwards).
    // No, this is not a mistake: Solarium does not generate valid JSON for
    // this, it is a single object with one "add" key for each document.
    /* @see \Solarium\QueryType\Update\RequestBuilder\Json::getRawData() */
    $limit = static::SEARCHSTAX_MAX_REQUEST_SIZE - 1;
    $doc_count_limit = isset($timeout_docs) ? (int) ($timeout_docs / 2) : count($docs);
    foreach ($docs as $doc) {
      $doc_json_size = strlen(json_encode($doc)) + 15;
      $current_batch_count = count($batch);
      if ($doc_json_size > $limit || $current_batch_count >= $doc_count_limit) {
        if (!$batch) {
          $item_id = $doc['ss_search_api_id'] ?? $doc['id'] ?? NULL;
          $with_id = isset($item_id) ? " with ID \"$item_id\"" : NULL;
          throw new SearchApiSolrException("Could not index item$with_id because its size exceeded SearchStax server's maximum request size.");
        }
        $new_update_query = $this->getUpdateQuery();
        $new_update_query->addDocuments($batch);
        try {
          $last_result = parent::update($new_update_query, $endpoint);
        }
        catch (SearchApiSolrException $e) {
          // Do not recurse more than five times, or if the remaining batch size
          // is too small.
          if ($current_batch_count <= 20 || $retry_level >= 4) {
            throw $e;
          }
          // In case this error was caused by a timeout, try again with a still
          // smaller batch size.
          $previous = $e->getPrevious();
          if (
            $previous instanceof HttpException
            && $previous->getCode() === CURLE_OPERATION_TIMEDOUT
          ) {
            return $this->updateFallback(
              $query,
              $endpoint,
              $current_batch_count,
              $retry_level + 1,
            );
          }
          throw $e;
        }
        $log_success($current_batch_count);
        $batch = [];
        $limit = static::SEARCHSTAX_MAX_REQUEST_SIZE - 1;
      }
      $limit -= $doc_json_size;
      $batch[] = $doc;
    }
    if ($batch) {
      $new_update_query = $this->getUpdateQuery();
      $new_update_query->addDocuments($batch);
      $last_result = parent::update($new_update_query, $endpoint);
      $log_success(count($batch));
    }

    return $last_result;
  }

  /**
   * {@inheritdoc}
   */
  public function getFile($file = NULL) {
    $server = $this->getServer();
    $server_id = $server->id();
    if (!isset(static::$cachedFiles[$server_id])) {
      $this->solrConfigSetController->setServer($server);
      $files = $this->solrConfigSetController->getConfigFiles();
      foreach ($files as $name => $content) {
        $content = preg_replace('/"drupal-\d+\.\d+\.\d+[^"]+"/m', '"' . $this->getSchemaVersionString() . '"', $content);
        $files[$name] = [
          'body' => $content,
          'size' => strlen($content),
        ];
      }
      ksort($files);
      static::$cachedFiles[$server_id] = $files;
    }

    if (!$file) {
      return new Response(json_encode([
        'files' => static::$cachedFiles[$server_id],
      ]));
    }
    if (empty(static::$cachedFiles[$server_id][$file])) {
      throw new SearchApiSolrException('File not found');
    }
    return new Response(static::$cachedFiles[$server_id][$file]['body']);
  }

  /**
   * {@inheritdoc}
   */
  public function alterConfigFiles(array &$files, string $lucene_match_version, string $server_id = ''): void {
    parent::alterConfigFiles($files, $lucene_match_version, $server_id);

    if (strpos($files['solrconfig.xml'], 'numVersionBuckets') === FALSE) {
      $files['solrconfig.xml'] = str_replace('</updateLog>', '<int name="numVersionBuckets">${solr.ulog.numVersionBuckets:65536}</int>' . "\n</updateLog>", $files['solrconfig.xml']);
    }
    $files['solrconfig.xml'] = str_replace('{solr.autoCommit.MaxTime:15000}', '{solr.autoCommit.MaxTime:600000}', $files['solrconfig.xml']);
    $files['solrconfig.xml'] = str_replace('{solr.autoSoftCommit.MaxTime:5000}', '{solr.autoSoftCommit.maxTime:300000}', $files['solrconfig.xml']);

    // Leverage the implicit Solr request handlers with default settings for
    // Solr Cloud.
    // @see https://lucene.apache.org/solr/guide/8_0/implicit-requesthandlers.html
    $files['solrconfig_extra.xml'] = preg_replace("@<requestHandler\s+name=\"/replication\".*?</requestHandler>@s", '', $files['solrconfig_extra.xml']);
    $files['solrconfig_extra.xml'] = preg_replace("@<requestHandler\s+name=\"/get\".*?</requestHandler>@s", '', $files['solrconfig_extra.xml']);

    // Set the StatsCache.
    // @see https://lucene.apache.org/solr/guide/8_0/distributed-requests.html#configuring-statscache-distributed-idf
    if (!empty($this->configuration['stats_cache'])) {
      $files['solrconfig_extra.xml'] .= '<statsCache class="' . $this->configuration['stats_cache'] . '" />' . "\n";
    }

    // solrcore.properties won’t work in SolrCloud mode (it is not read from
    // ZooKeeper). Therefore, we go for a more specific fallback to keep the
    // possibility to set the property as parameter of the virtual machine.
    // @see https://lucene.apache.org/solr/guide/8_6/configuring-solrconfig-xml.html
    $files['solrconfig.xml'] = preg_replace('/solr.luceneMatchVersion:LUCENE_\d+/', 'solr.luceneMatchVersion:' . $this->getLuceneMatchVersion(), $files['solrconfig.xml']);
    unset($files['solrcore.properties']);
  }

  /**
   * Retrieves the server to which this connector plugin is linked.
   *
   * @return \Drupal\search_api\ServerInterface
   *   The server.
   *
   * @throws \Drupal\search_api_solr\SearchApiSolrException
   *   Thrown if the server could not be determined.
   */
  protected function getServer(): ServerInterface {
    // First try to retrieve the server from the call stack.
    if (!isset($this->server)) {
      $this->setServerFromBacktrace();
    }
    // If that doesn't work, the only remaining approach is to load all
    // SearchStax servers and match against their connector configs.
    if (!isset($this->server)) {
      try {
        $server_storage = $this->entityTypeManager->getStorage('search_api_server');
        $server_ids = $server_storage->getQuery()
          ->condition('backend', 'search_api_solr')
          ->condition('backend_config.connector', 'searchstax')
          ->execute();
        if ($server_ids) {
          /** @var \Drupal\search_api\ServerInterface $server */
          foreach ($server_storage->loadMultiple($server_ids) as $server) {
            if (($server->getBackendConfig()['connector_config'] ?? []) === $this->configuration) {
              $this->server = $server;
              break;
            }
          }
        }
      }
      catch (\Exception $ignored) {
      }
    }
    // If that didn't work, either, we can only throw an exception to at least
    // prevent a fatal error.
    if (!isset($this->server)) {
      throw new SearchApiSolrException('Could not determine server for connector plugin.');
    }
    return $this->server;
  }

  /**
   * Attempts to extract the plugin's server from the backtrace.
   */
  protected function setServerFromBacktrace(): void {
    $options = DEBUG_BACKTRACE_PROVIDE_OBJECT | DEBUG_BACKTRACE_IGNORE_ARGS;
    $limit = 8;
    $trace = debug_backtrace($options, $limit);
    foreach ($trace as $call) {
      $object = $call['object'] ?? NULL;
      if ($object instanceof ServerInterface) {
        $this->server = $object;
        break;
      }
      if ($object instanceof BackendInterface) {
        $this->server = $object->getServer();
        break;
      }
    }
  }

}
