<?php

declare(strict_types=1);

namespace Drupal\searchstax\Hook;

use Drupal\Core\Entity\EntityStorageException;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\searchstax\Contrib\MigrateToKeysInterface;

/**
 * Provides hook implementations and helper methods for moving config to keys.
 */
class MigrateToKeys {

  use StringTranslationTrait;

  /**
   * The "migrate to keys" service.
   */
  protected MigrateToKeysInterface $migrateToKeys;

  /**
   * The messenger service.
   */
  protected MessengerInterface $messenger;

  public function __construct(
    MigrateToKeysInterface $migrate_to_keys,
    MessengerInterface $messenger,
    TranslationInterface $string_translation
  ) {
    $this->migrateToKeys = $migrate_to_keys;
    $this->messenger = $messenger;
    $this->stringTranslation = $string_translation;
  }

  /**
   * Reacts to modules being installed.
   *
   * Moves all sensitive configuration for this module to new keys when the Key
   * module is installed.
   */
  #[Hook('modules_installed')]
  public function onModulesInstalled(array $modules, bool $is_syncing): void {
    // React to the Key module being installed.
    if ($is_syncing || !in_array('key', $modules)) {
      return;
    }

    try {
      // Import default keys and migrate existing servers.
      $this->migrateToKeys->ensureDefaultKeys();
      // Migrate analytics credentials.
      $this->migrateToKeys->migrateAnalyticsCredentials();
      // Migrate server connector credentials.
      $migrated_count = $this->migrateToKeys->migrateServersToKeys();

      if ($migrated_count > 0) {
        $message = $this->formatPlural(
          $migrated_count,
          'The Key module was installed. Successfully migrated SearchStax connector credentials for @count server to use Key-based storage.',
          'The Key module was installed. Successfully migrated SearchStax connector credentials for @count servers to use Key-based storage.',
        );
        $this->messenger->addStatus($message);
      }
    }
    catch (EntityStorageException $e) {
      $this->messenger->addError($this->t('The Key module was installed, but migrating SearchStax configuration to use Key-based storage failed with %exception_class: %message', [
        '%exception_class' => get_class($e),
        '%message' => $e->getMessage(),
      ]));
    }
  }

}
