<?php

namespace Drupal\searchstax\Hook;

use Drupal\Core\DependencyInjection\DependencySerializationTrait;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Form\SubformState;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Messenger\MessengerTrait;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\search_api\Plugin\views\query\SearchApiQuery;
use Drupal\search_api\SearchApiException;
use Drupal\searchstax\Exception\NotLoggedInException;
use Drupal\searchstax\Exception\SearchStaxException;
use Drupal\searchstax\Form\ApiLoginFormTrait;
use Drupal\searchstax\Service\ApiInterface;
use Drupal\searchstax\Service\SearchStaxServiceInterface;
use Drupal\searchstax\Service\VersionCheckInterface;
use Drupal\views\ViewEntityInterface;

/**
 * Provides admins with the option to select relevance models for search views.
 */
class ViewsSelectRelevanceModel {

  use ApiLoginFormTrait {
    showLogInForm as showLogInFormTrait;
    validateLoginForm as validateLoginFormTrait;
    submitLoginForm as submitLoginFormTrait;
  }
  use DependencySerializationTrait;
  use MessengerTrait;
  use StringTranslationTrait;

  /**
   * The SearchStax utility service.
   */
  protected SearchStaxServiceInterface $utility;

  /**
   * The SearchStax version check service.
   */
  protected VersionCheckInterface $versionCheck;

  /**
   * The language manager.
   */
  protected LanguageManagerInterface $languageManager;

  /**
   * Constructs a new class instance.
   *
   * @param \Drupal\searchstax\Service\ApiInterface $searchStaxApi
   *   The SearchStax API service.
   * @param \Drupal\searchstax\Service\SearchStaxServiceInterface $utility
   *   The SearchStax utility service.
   * @param \Drupal\searchstax\Service\VersionCheckInterface $versionCheck
   *   The SearchStax version check service.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   * @param \Drupal\Core\StringTranslation\TranslationInterface $stringTranslation
   *   The string translation service.
   * @param \Drupal\Core\Language\LanguageManagerInterface $languageManager
   *   The language manager.
   */
  public function __construct(
    ApiInterface $searchStaxApi,
    SearchStaxServiceInterface $utility,
    VersionCheckInterface $versionCheck,
    MessengerInterface $messenger,
    TranslationInterface $stringTranslation,
    LanguageManagerInterface $languageManager
  ) {
    $this->searchStaxApi = $searchStaxApi;
    $this->utility = $utility;
    $this->versionCheck = $versionCheck;
    $this->messenger = $messenger;
    $this->stringTranslation = $stringTranslation;
    $this->languageManager = $languageManager;
  }

  /**
   * Retrieves the relevance model to use for the given view and language.
   *
   * @param \Drupal\views\ViewEntityInterface $view
   *   The view in question.
   * @param string $langcode
   *   The code of the current language.
   *
   * @return string
   *   The relevance model selected for the view and language. An empty string
   *   if no model should be explicitly passed.
   */
  public static function getViewRelevanceModel(ViewEntityInterface $view, string $langcode): string {
    return $view->getThirdPartySettings('searchstax')['relevance_models'][$langcode] ?? '';
  }

  /**
   * Implements hook_form_FORM_ID_alter() for form "views_ui_edit_display_form".
   *
   * Alters the Views query settings form for views based on SearchStax servers.
   */
  #[Hook('form_views_ui_edit_display_form_alter')]
  public function alterForm(&$form, FormStateInterface $form_state): void {
    if ($form_state->get('section') !== 'query') {
      return;
    }

    /** @var \Drupal\views_ui\ViewUI $view */
    $view = $form_state->get('view');

    // Determine whether this view is linked to a SearchStax server.
    $query = $view->getExecutable()->getQuery();
    if (!($query instanceof SearchApiQuery)) {
      return;
    }
    try {
      $server = $query->getIndex()->getServerInstance();
    }
    catch (SearchApiException $ignored) {
      return;
    }
    if (!$this->utility->isSearchStaxSolrServer($server)) {
      return;
    }

    $subform = &$form['options']['query']['options']['searchstax'];
    $subform = [
      '#type' => 'fieldset',
      '#title' => $this->t('SearchStax settings'),
    ];
    if (!$this->searchStaxApi->isLoggedIn()) {
      $this->showLogInForm($subform, $form_state);
      return;
    }
    $subform['messages'] = [
      '#type' => 'status_messages',
      '#weight' => -10,
    ];
    try {
      $app_info = $this->versionCheck->getAppInformation($server);
      if (!$app_info) {
        throw new SearchStaxException("Could not determine SearchStax app for server \"{$server->id()}\"");
      }
      foreach ($this->languageManager->getLanguages() as $langcode => $language) {
        $language_name = $language->getName();
        try {
          $models = $this->searchStaxApi->getRelevanceModels(
            $app_info->getAccount(),
            $app_info->getAppId(),
            $langcode,
          );
        }
        catch (SearchStaxException $e) {
          // Just capture the case that the current language is not configured
          // in the SearchStax app.
          if (
            $e instanceof NotLoggedInException
            || substr($e->getMessage(), 0, 44) !== 'Results configuration does not exist for App'
          ) {
            throw $e;
          }
          $subform['relevance_models'][$langcode] = [
            '#type' => 'item',
            '#title' => $this->t('Relevance model for %language.', [
              '%language' => $language_name,
            ]),
            '#description' => $this->t('No relevance models are available for %language in this SearchStax app.', [
              '%language' => $language_name,
            ]),
          ];
          continue;
        }
        $options = [
          '' => $this->t('- Use default -'),
        ];
        foreach ($models as $model) {
          if ($model['status'] !== 'published') {
            continue;
          }
          $name = $model['name'];
          if (!empty($model['default'])) {
            $name = $this->t('@model (default)', ['@model' => $name]);
          }
          $options[$model['id']] = $name;
        }
        $subform['relevance_models'][$langcode] = [
          '#type' => 'radios',
          '#title' => $this->t('Relevance model for %language.', [
            '%language' => $language_name,
          ]),
          '#description' => $this->t('The relevance model that should be used for %language searches with this view.', [
            '%language' => $language_name,
          ]),
          '#options' => $options,
          '#default_value' => $this->getViewRelevanceModel($view, $langcode),
        ];
      }
    }
    catch (NotLoggedInException $e) {
      $this->showLogInForm($subform, $form_state);
      return;
    }
    catch (SearchStaxException $e) {
      unset($form['options']['query']['options']['searchstax']);
      $this->messenger()->addError($this->t('Error while retrieving available relevance models from SearchStax server: @message.', [
        '@message' => $e->getMessage(),
      ]));
      return;
    }
    array_unshift($form['actions']['submit']['#submit'], [$this, 'submitViewsQueryForm']);
  }

  /**
   * Submit handler for our additions to the Views query form.
   */
  public function submitViewsQueryForm(array &$form, FormStateInterface $form_state): void {
    if ($form_state->get('section') !== 'query') {
      return;
    }

    /** @var \Drupal\views_ui\ViewUI $view */
    $view = $form_state->get('view');
    $models = $form_state->getValue(['query', 'options', 'searchstax', 'relevance_models'], '');
    $form_state->unsetValue(['query', 'options', 'searchstax']);
    $view->setThirdPartySetting('searchstax', 'relevance_models', $models);
  }

  /**
   * {@inheritdoc}
   */
  protected function showLogInForm(&$form, FormStateInterface $form_state): void {
    $form['message'] = [
      '#theme' => 'status_messages',
      '#message_list' => [
        'warning' => [
          t('You need to log in to the SearchStax app in order to change these settings.'),
        ],
      ],
      '#status_headings' => [
        'status' => t('Status message'),
        'error' => t('Error message'),
        'warning' => t('Warning message'),
      ],
    ];
    $form = $this->showLogInFormTrait($form, $form_state);

    $html_id = 'searchstax-login-form';
    $form['#attributes']['id'] = $html_id;
    $form['login']['#type'] = 'details';
    $submit_button = $form['actions']['submit'];
    unset($form['actions']);
    assert(isset($submit_button['#validate']));
    $submit_button['#validate'] = [[$this, 'validateLoginForm']];
    assert(isset($submit_button['#submit']));
    $submit_button['#submit'] = [[$this, 'submitLoginForm']];
    $submit_button['#ajax'] = [
      'callback' => [static::class, 'buildConfigForm'],
      'wrapper' => $html_id,
      'method' => 'replaceWith',
      'effect' => 'fade',
    ];
    $form['submit'] = $submit_button;
  }

  /**
   * {@inheritdoc}
   */
  public function validateLoginForm(array &$form, FormStateInterface $form_state): void {
    $subform = &$form['options']['query']['options']['searchstax']['login'];
    $subform_state = SubformState::createForSubform($subform, $form, $form_state);
    $this->validateLoginFormTrait($subform, $subform_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitLoginForm(array &$form, FormStateInterface $form_state): void {
    $this->submitLoginFormTrait($form, $form_state);
    $form_state->setRebuild();
  }

  /**
   * Handles switching the selected backend plugin.
   *
   * @param array $form
   *   The current form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current form state.
   *
   * @return array
   *   The part of the form to return as AJAX.
   */
  public static function buildConfigForm(array $form, FormStateInterface $form_state): array {
    return $form['options']['query']['options']['searchstax'];
  }

}
